* compud.dta is a Stata file downloaded via WRDS from the Compustat North America Security Daily dataset
* variables (in addition to those that come automatically): TIC EXCHG TPCI GSUBIND AJEXDI CSHOC PRCCD TRFD
* Dates: 1/1/2016 to 1/1/2018

* compua.dta is a Stata file downloaded via WRDS from the Compustat North America Fundamentals Annual File
* variables: TIC CEQ PI TXPD TXT 
* Dates: 1/2015 to 5/2017

* Also uses vangetfs.dta, gsector.dta, ggroup.dta, gind.dta, gsubind.dta, trump_departure.dta reelect.dta approval.dta ffd.dta, ff5d.dta

* Requires installation of reghdfe command (for generating double-clustered standard errors)

local taxrates = 1
local returns = 1
local indices = 1
local table1 = 1
local table2 = 1
local tableA1 = 1
local graphs = 1

local zeroday = mdy(11,8,2016)

set more off
clear
cap log close

/* Read Compua.dta and generate tax rate file */
if `taxrates' == 1 {
	use compua, clear
	keep if indfmt=="INDL"
	keep if curcd=="USD"
	keep gvkey datadate pi txt txpd ceq
	gen tax_rate_paid = txpd/pi if pi>0
	gen tax_rate_total = txt/pi if pi>0

	/* Keep most recent annual data, so long as dated June 30 2016 or earlier; if none available keep more recent one */  
	replace datadate=datadate-10000 if datadate>mdy(6,30,2016)
	sort gvkey datadate
	qby gvkey: keep if _n==_N
	replace datadate=datadate+10000 if datadate<mdy(1,1,2015)

	rename datadate date_tax
	compress
	sort gvkey
	save taxrates, replace
	}

/* Read compud.dta and generate returns file */	
if `returns' == 1 {	
	use compud, clear
	rename datadate d

	/* Drop Canadian/non-US stocks */
	drop if exch>=7&exch<=10
	drop if exch==1

	/* Drop market holidays; days with incomplete data */
	drop if d==mdy(1,18,2016)|d==mdy(5,30,2016)|d==mdy(7,4,2016)|d==mdy(9,5,2016)|d==mdy(11,24,2016)|d==mdy(1,16,2017)|d==mdy(5,29,2017)|d==mdy(7,4,2017)|d==mdy(9,4,2017)|d==mdy(11,23,2017)
	drop if d>mdy(1,1,2018)

	/* Merge with list of Vanguard style ETFs so they do not get dropped */
	sort tic
	merge tic using vangetfs, nokeep
	replace tpci="00" if _merge==3
	drop _merge

	/* Drop preferreds, warrants, units, convertible preferreds, structured products, open-ended mutual funds, and GSAs*/
	drop if tpci=="+"|tpci=="1"|tpci=="2"|tpci=="4"|tpci=="G"|tpci=="R"|tpci=="S"

	/* Drop ETFs, CEFs, ADRs */
	drop if tpci=="%"|tpci=="7"|tpci=="F"

	/* Keep only listed (NYSE/AMEX/NASDAQ) common stocks */
	keep if tpci=="0"|tpci=="00"
	keep if exchg==11|exchg==14|exchg==12|(exchg==17&tpci=="00")

	/* Merge with GICS names */
	gen gsector=substr(gsubind,1,2)
	gen ggroup=substr(gsubind,1,4)
	gen gind=substr(gsubind,1,6)
	
	sort gsector
	merge gsector using gsector, nokeep
	drop _merge

	sort ggroup
	merge ggroup using ggroup, nokeep
	drop _merge

	sort gind
	merge gind using gind, nokeep
	drop _merge

	sort gsubind
	merge gsubind using gsubind, nokeep
	drop _merge

	gen gsec=gsector+" "+gsectorname
	gen ggrp=ggroup+" "+ggroupname
	gen gin=gind+" "+gindname
	gen gsin=gsubind+" "+gsubindname

	/* If dividend gross-up factor missing assume no dividends */
	egen mtrfd=mean(trfd), by(gvkey iid)
	replace trfd=1 if mtrfd==.
	drop mtrfd
	
	/* Create return and prior-day market-cap */
	sort gvkey iid d
	qby gvkey iid: gen mcap=cshoc[_n-1]*prccd[_n-1]
	qby gvkey iid: gen ret=(trfd*prccd/ajexdi)/(trfd[_n-1]*prccd[_n-1]/ajexdi[_n-1])-1

	/* Create variables for style returns */
	foreach s in MKT LB LG LV MB MG MV SB SG SV {
		gen temp=ret if style=="`s'"
		egen ret`s' = mean(temp), by(d)
		drop temp
		}
		
	sort gvkey iid d
	compress
	save returns, replace
	}

/* Calculate indices (unfinished) */
if `indices' == 1 {
	use returns, clear
	
	sort gvkey
	merge gvkey using taxrates, nokeep
	drop _merge
	
	rename tax_rate_paid tax_paid
	rename tax_rate_total tax_total
	
	keep ret tax_* d gvkey iid mcap gsector ggroup gind gsubind retL* retM* retS*
	
	gen retmcap=ret*mcap
	rename ret ret_stock
		
	foreach v of varlist tax_total tax_paid {
		replace `v' = 0 if `v'<0
		replace `v' = . if `v'>.5&`v'~=.
		}	
	
	/* Create 4 levels of GICS industry returns */
	foreach v of varlist gsector ggroup gind gsubind {
		egen temp1 = sum(retmcap), by(`v' d)
		egen temp2 = sum(mcap) if retmcap~=., by(`v' d)
		gen temp3 = temp1/temp2
		egen ret_`v' = mean(temp3), by(`v' d)
		drop temp1 temp2 temp3
		}	
	
	
	/* Create variables for returns on Nov 9 2016 */
	foreach v of varlist ret_stock retL* retM* retS* {
		gen temp=`v' if d==mdy(11,9,2016)
		egen `v'_Nov9 = mean(temp), by(gvkey iid)
		drop temp
		}
	
	foreach v in gsector ggroup gind gsubind  {	
		gen temp=ret_`v' if d==mdy(11,9,2016)
		egen ret_`v'_Nov9 = mean(temp), by(`v')
		drop temp
		}		

		
	/* Create variable for Nov 9 market cap */
	gen temp=mcap if d==mdy(11,9,2016)
	egen mcap_Nov9 = mean(temp), by(gvkey iid)
	drop temp
	
	/* Create variables for Nov 9 returns explained by and orthogonal to variables */
	foreach v of varlist tax_total tax_paid { 
		reg ret_stock_Nov9 `v' [aw=mcap_Nov9]
		predict w_pred_`v'
		predict w_resid_`v', resid
		}
		
	foreach v of varlist gsector ggroup gind gsubind  { 
		areg ret_stock_Nov9 [aw=mcap_Nov9], absorb(`v')
		predict w_pred_`v', xbd
		predict w_resid_`v', r
		}
		

	/* Create index weights and returns based on Nov 9 market-adjusted returns */	
	foreach v in stock gsector ggroup gind gsubind {
			
		gen wl`v'=mcap*max(0,ret_`v'_Nov9-retMKT_Nov9)
		egen sumw=sum(wl`v'), by(d)
		replace wl`v'=wl`v'/sumw
		drop sumw
		gen retw=(ret_stock-retMKT)*wl`v'
		egen temp1=sum(retw), by(d)
		egen temp2=sum(wl`v') if retw~=., by(d)
		gen temp3=temp1/temp2
		egen retl_`v'=mean(temp3), by(d)
		drop temp1 temp2 temp3 retw
		
		gen ws`v'=mcap*max(0,-1*(ret_`v'_Nov9-retMKT_Nov9))
		egen sumw=sum(ws`v'), by(d)
		replace ws`v'=ws`v'/sumw
		drop sumw
		gen retw=(ret_stock-retMKT)*ws`v'
		egen temp1=sum(retw), by(d)
		egen temp2=sum(ws`v') if retw~=., by(d)
		gen temp3=temp1/temp2
		egen rets_`v'=mean(temp3), by(d)
		drop temp1 temp2 temp3 retw
		}
		
	/* Create index weights and returns based on explained and orthogonal returns */
	foreach v of varlist tax_paid tax_total gsector ggroup gind gsubind {
		foreach n in pred resid {
			su w_`n'_`v' [aw=mcap] if d==mdy(11,9,2016)
			local m=r(mean)
			gen wl_`n'_`v'=mcap*max(0,w_`n'_`v'-`m')
			egen sumw=sum(wl_`n'_`v'), by(d)
			replace wl_`n'_`v'=wl_`n'_`v'/sumw
			drop sumw
			gen retw=(ret_stock-retMKT)*wl_`n'_`v'
			egen temp1=sum(retw), by(d)
			egen temp2=sum(wl_`n'_`v') if retw~=., by(d)
			gen temp3=temp1/temp2
			egen retl_`n'_`v'=mean(temp3), by(d)
			drop temp1 temp2 temp3 retw
			
			gen ws_`n'_`v'=mcap*min(0,w_`n'_`v'-`m')
			egen sumw=sum(ws_`n'_`v'), by(d)
			replace ws_`n'_`v'=ws_`n'_`v'/sumw
			drop sumw
			gen retw=(ret_stock-retMKT)*ws_`n'_`v'
			egen temp1=sum(retw), by(d)
			egen temp2=sum(ws_`n'_`v') if retw~=., by(d)
			gen temp3=temp1/temp2
			egen rets_`n'_`v'=mean(temp3), by(d)
			drop temp1 temp2 temp3 retw
			}
		}
		
		
	/* Create long-short indices */	
	foreach v in stock gsector ggroup gind gsubind {

		sort d	
		gen indexl_`v' = 100 if d==`zeroday'
		replace indexl_`v' = indexl_`v'[_n-1]*(1+retl_`v'*(d>d[_n-1])) if indexl_`v'[_n-1]~=.
		gen indexs_`v' = 100 if d==`zeroday'
		replace indexs_`v' = indexs_`v'[_n-1]*(1+rets_`v'*(d>d[_n-1])) if indexs_`v'[_n-1]~=.
		gsort -d
		replace indexl_`v' = indexl_`v'[_n-1]/(1+retl_`v'[_n-1]*(d<d[_n-1])) if indexl_`v'[_n-1]~=.
		replace indexs_`v' = indexs_`v'[_n-1]/(1+rets_`v'[_n-1]*(d<d[_n-1])) if indexs_`v'[_n-1]~=.	
		gen ls_`v' = indexl_`v'-indexs_`v'
		}

		
	foreach v of varlist tax_paid tax_total gsector ggroup gind gsubind {
		foreach n in pred resid {
			sort d	
			gen indexl_`n'_`v' = 100 if d==`zeroday'
			replace indexl_`n'_`v' = indexl_`n'_`v'[_n-1]*(1+retl_`n'_`v'*(d>d[_n-1])) if indexl_`n'_`v'[_n-1]~=.
			gen indexs_`n'_`v' = 100 if d==`zeroday'
			replace indexs_`n'_`v' = indexs_`n'_`v'[_n-1]*(1+rets_`n'_`v'*(d>d[_n-1])) if indexs_`n'_`v'[_n-1]~=.
			gsort -d
			replace indexl_`n'_`v' = indexl_`n'_`v'[_n-1]/(1+retl_`n'_`v'[_n-1]*(d<d[_n-1])) if indexl_`n'_`v'[_n-1]~=.
			replace indexs_`n'_`v' = indexs_`n'_`v'[_n-1]/(1+rets_`n'_`v'[_n-1]*(d<d[_n-1])) if indexs_`n'_`v'[_n-1]~=.	
			gen ls_`n'_`v' = indexl_`n'_`v'-indexs_`n'_`v'
			}
		}		
		
	save indices_full2, replace
	
	keep d ls_* indexl_* indexs_*
	collapse (mean) ls_* index*, by(d)
	
	save indices2, replace
	}
	
/* Make tables for paper */
if `table1' == 1 {

	use returns, clear
	
	gen lret=ln(1+ret) if d<mdy(11,9,2016)
	
	egen pastret=sum(lret), by(gvkey iid)
	egen ctret=count(lret), by(gvkey iid)

	replace pastret=. if ctret<215
	keep if d==mdy(11,9,2016)
	
	sort gvkey
	merge gvkey using taxrates, nokeep
	drop _merge
	
	rename tax_rate_paid tax_paid
	rename tax_rate_total tax_total
	
	gen bm=ceq/(cshoc*prccd)*1000000
		
	foreach v of varlist tax_total tax_paid {
		replace `v' = 0 if `v'<0
		replace `v' = . if `v'>.5&`v'~=.
		}	
	
	keep ret tax_* d gvkey iid mcap gsector ggroup gind gsubind gsec ggrp gin gsin retMKT tic pastret ctret bm ceq
	
	rename ret ret_stock
	
	/* Create variable for Nov 9 returns */
	foreach v of varlist ret_stock retMKT {
		gen temp=`v' if d==mdy(11,9,2016)
		egen `v'_Nov9 = mean(temp), by(gvkey iid)
		drop temp
		}		
		
	/* Create variable for Nov 9 market cap */
	gen temp=mcap if d==mdy(11,9,2016)
	egen mcap_Nov9 = mean(temp), by(gvkey iid)
	drop temp
	
	gen w = mcap*(ret_stock_Nov9-retMKT_Nov9) if d==mdy(11,9,2016)
	
	/* Create variables for Nov 9 returns explained by and orthogonal to variables */
	foreach v of varlist tax_paid  { 
		reg ret_stock_Nov9 `v' [aw=mcap_Nov9]
		predict w_pred_`v'
		predict w_resid_`v', resid
		}
		
	foreach v of varlist gsubind  { 
		areg ret_stock_Nov9 [aw=mcap_Nov9], absorb(`v')
		predict w_pred_`v', xbd
		predict w_resid_`v', r
		}
		
	foreach v of varlist w_* {
		su `v' [aw=mcap]
		local m=r(mean)
		replace `v'=(`v'-r(mean))*mcap
		}

	gen cat = gsec
	replace cat = ggrp if gsector=="35"
	
	gen cat2=gsin if gsubind=="10102050"|gsubind=="25401030"|gsubind=="25302010"|gsubind=="55105020"
	replace cat2="Prisons" if tic=="GEO"|tic=="CXW"
		
	/* Table 1 */
	
	log using table1, t replace

	foreach v of varlist w* {
		gen abs`v'=abs(`v')
		gen port_`v'=(`v'>0) if `v'~=.
		table port_`v', c(freq sum mcap)
		table port_`v' [aw=abs`v'], c(mean mcap median mcap mean ret_stock_Nov9 mean tax_paid mean tax_total)
		table port_`v' [aw=abs`v'], c(mean bm mean pastret)
		
		table cat port_`v', c(sum abs`v')
		
		table cat2 port_`v', c(sum abs`v')
		}
	log close
		
	}
		
if `table2' == 1 {

	use returns, clear
	
	keep gvkey iid d mcap ret retMKT gsector ggroup gind gsubind 
	drop if d<mdy(11,8,2016)
	drop if d>mdy(12,31,2017)
	drop if mcap==.
	
	sort gvkey
	merge gvkey using taxrates, nokeep
	drop _merge
	
	rename tax_rate_paid tax_paid
	rename tax_rate_total tax_total
	
	foreach v of varlist tax_total tax_paid {
		replace `v' = 0 if `v'<0
		replace `v' = . if `v'>.5&`v'~=.
		}
	
	gen temp=mcap if d==mdy(11,9,2016)
	egen mcap_Nov9=mean(temp), by(gvkey iid)
	drop temp

	gen temp=(ret-retMKT) if d==mdy(11,9,2016)
	egen ret_Nov9=mean(temp), by(gvkey iid)
	drop temp	
	
	/* Create variables for Nov 9 returns explained by and orthogonal to variables */
	foreach v of varlist tax_total tax_paid { 
		reg ret_Nov9 `v' [aw=mcap_Nov9]
		predict w_pred_`v'
		predict w_resid_`v', resid
		}
		
	foreach v of varlist gsubind  { 
		areg ret_Nov9 [aw=mcap_Nov9], absorb(`v')
		predict w_pred_`v', xbd
		predict w_resid_`v', r
		}
		
	rename ret_Nov9 w_overall

	sort d
	merge d using ff5d, nokeep
	drop _merge
	rename smb smb5
	
	sort d
	merge d using ffd, nokeep
	drop _merge
	
	rename mktrf factor1
	rename smb factor2
	rename hml factor3
	rename umd factor4
	rename rmw factor5
	rename cma factor6
	rename smb5 factor7
	
	save table2_temp, replace


	use table2_temp, clear
	
	gen byte one = 1
	
	/* Window lengths:  1 = 13 days; 2 = 21; 3 = 2; 4 = 1; 5 = 3; 6 = 2; 7 = 5  */
	gen event = .
	replace event = 0 if d>mdy(11,8,2016)&d<=mdy(11,9,2016)	
	replace event = 1 if d>mdy(3,7,2017)&d<=mdy(3,24,2017)
	replace event = 2 if d>mdy(5,8,2017)&d<=mdy(6,7,2017)
	replace event = 3 if d>mdy(6,7,2017)&d<=mdy(6,9,2017)
	replace event = 4 if d>mdy(7,24,2017)&d<=mdy(7,25,2017)
	replace event = 5 if d>mdy(7,25,2017)&d<=mdy(7,28,2017)
	replace event = 6 if d>mdy(10,26,2017)&d<=mdy(10,30,2017)
	replace event = 7 if d>mdy(11,27,2017)&d<=mdy(12,4,2017)		
	
	egen byte gg=group(ggroup)
	
	gen index=.
	gen co=.
	gen se=.
	gen mult=.
	gen ev=.
	gen model=.
	
	forvalues k=1/7 {
		gen beta`k'=.
		gen sebeta`k'=.
		}
	
	local m=1
	local j=1
	
	foreach n in _overall _pred_gsubind _resid_gsubind _pred_tax_paid _resid_tax_paid {
		quietly su ret if event==0&w`n'>0 [aw=mcap*w`n']
		local m1 = r(mean)
		quietly su ret if event==0&w`n'<0 [aw=-1*mcap*w`n']
		local m2 = r(mean)
		local mdiff = int((`m1' - `m2')*10000)/100
		
		forvalues i=1/7 {
			gen w`n'_event`i' = w`n'*(event==`i')
			}
			
		forvalues k=1/7 {
			gen w`n'_factor`k' = w`n'*factor`k'
			}
		
		/* Raw returns */
		reghdfe ret w`n' w`n'_event* [aw=mcap], absorb(d) vce(cluster gg d)
		forvalues i=1/7 {
			lincom w`n'+w`n'_event`i'
			replace co=r(estimate) in `m'
			replace se=r(se) in `m'
			replace mult=`mdiff' in `m'
			replace ev=`i' in `m'
			replace index=`j' in `m'
			replace model=0 in `m'
			local m=`m'+1
			}
		lincom w`n'_event3*2+w`n'_event4+w`n'_event7*5-w`n'_event1*13-w`n'_event2*21-w`n'_event5*3-w`n'_event6*2	
		replace co=r(estimate) in `m'
		replace se=r(se) in `m'
		replace mult=`mdiff' in `m'
		replace ev=8 in `m'
		replace index=`j' in `m'
		replace model=0 in `m'
		local m=`m'+1
			
			
		/* CAPM model */
		reghdfe ret w`n' w`n'_event* w`n'_factor1 [aw=mcap], absorb(d) vce(cluster gg d)
		forvalues i=1/7 {
			forvalues k=1/1 {
				replace beta`k'=_coef[w`n'_factor`k'] in `m'
				replace sebeta`k'=_se[w`n'_factor`k'] in `m'
				}
			lincom w`n'+w`n'_event`i'
			replace co=r(estimate) in `m'
			replace se=r(se) in `m'
			replace mult=`mdiff' in `m'
			replace ev=`i' in `m'
			replace index=`j' in `m'
			replace model=1 in `m'

			local m=`m'+1
			}			
		lincom w`n'_event3*2+w`n'_event4+w`n'_event7*5-w`n'_event1*13-w`n'_event2*21-w`n'_event5*3-w`n'_event6*2	
		replace co=r(estimate) in `m'
		replace se=r(se) in `m'
		replace mult=`mdiff' in `m'
		replace ev=8 in `m'
		replace index=`j' in `m'
		replace model=1 in `m'
		local m=`m'+1
		
		/* CAPM+SMB model */
		reghdfe ret w`n' w`n'_event* w`n'_factor1 w`n'_factor2 [aw=mcap], absorb(d) vce(cluster gg d)
		forvalues i=1/7 {
			forvalues k=1/2 {
				replace beta`k'=_coef[w`n'_factor`k'] in `m'
				replace sebeta`k'=_se[w`n'_factor`k'] in `m'
				}
			lincom w`n'+w`n'_event`i'
			replace co=r(estimate) in `m'
			replace se=r(se) in `m'
			replace mult=`mdiff' in `m'
			replace ev=`i' in `m'
			replace index=`j' in `m'
			replace model=2 in `m'

			local m=`m'+1
			}			
		lincom w`n'_event3*2+w`n'_event4+w`n'_event7*5-w`n'_event1*13-w`n'_event2*21-w`n'_event5*3-w`n'_event6*2	
		replace co=r(estimate) in `m'
		replace se=r(se) in `m'
		replace mult=`mdiff' in `m'
		replace ev=8 in `m'
		replace index=`j' in `m'
		replace model=2 in `m'
		local m=`m'+1
		
		/* FF3 model */
		reghdfe ret w`n' w`n'_event* w`n'_factor1 w`n'_factor2 w`n'_factor3 [aw=mcap], absorb(d) vce(cluster gg d)
		forvalues i=1/7 {
			forvalues k=1/3 {
				replace beta`k'=_coef[w`n'_factor`k'] in `m'
				replace sebeta`k'=_se[w`n'_factor`k'] in `m'
				}
			lincom w`n'+w`n'_event`i'
			replace co=r(estimate) in `m'
			replace se=r(se) in `m'
			replace mult=`mdiff' in `m'
			replace ev=`i' in `m'
			replace index=`j' in `m'
			replace model=3 in `m'

			local m=`m'+1
			}
		lincom w`n'_event3*2+w`n'_event4+w`n'_event7*5-w`n'_event1*13-w`n'_event2*21-w`n'_event5*3-w`n'_event6*2	
		replace co=r(estimate) in `m'
		replace se=r(se) in `m'
		replace mult=`mdiff' in `m'
		replace ev=8 in `m'
		replace index=`j' in `m'
		replace model=3 in `m'
		local m=`m'+1
		
		/* Carhart model */
		reghdfe ret w`n' w`n'_event* w`n'_factor1 w`n'_factor2 w`n'_factor3 w`n'_factor4 [aw=mcap], absorb(d) vce(cluster gg d)
		forvalues i=1/7 {
			forvalues k=1/4 {
				replace beta`k'=_coef[w`n'_factor`k'] in `m'
				replace sebeta`k'=_se[w`n'_factor`k'] in `m'
				}
			lincom w`n'+w`n'_event`i'
			replace co=r(estimate) in `m'
			replace se=r(se) in `m'
			replace mult=`mdiff' in `m'
			replace ev=`i' in `m'
			replace index=`j' in `m'
			replace model=4 in `m'

			local m=`m'+1
			}			
		lincom w`n'_event3*2+w`n'_event4+w`n'_event7*5-w`n'_event1*13-w`n'_event2*21-w`n'_event5*3-w`n'_event6*2	
		replace co=r(estimate) in `m'
		replace se=r(se) in `m'
		replace mult=`mdiff' in `m'
		replace ev=8 in `m'
		replace index=`j' in `m'
		replace model=4 in `m'
		local m=`m'+1
		
		/* FF5 model */
		reghdfe ret w`n' w`n'_event* w`n'_factor1 w`n'_factor7 w`n'_factor3 w`n'_factor5 w`n'_factor6 [aw=mcap], absorb(d) vce(cluster gg d)
		forvalues i=1/7 {
			forvalues k=1/7 {
				cap replace beta`k'=_coef[w`n'_factor`k'] in `m'
				cap replace sebeta`k'=_se[w`n'_factor`k'] in `m'
				}
			lincom w`n'+w`n'_event`i'
			replace co=r(estimate) in `m'
			replace se=r(se) in `m'
			replace mult=`mdiff' in `m'
			replace ev=`i' in `m'
			replace index=`j' in `m'
			replace model=5 in `m'

			local m=`m'+1
			}			
		lincom w`n'_event3*2+w`n'_event4+w`n'_event7*5-w`n'_event1*13-w`n'_event2*21-w`n'_event5*3-w`n'_event6*2	
		replace co=r(estimate) in `m'
		replace se=r(se) in `m'
		replace mult=`mdiff' in `m'
		replace ev=8 in `m'
		replace index=`j' in `m'
		replace model=5 in `m'
		local m=`m'+1
		
		/* FF5+UMD model */
		reghdfe ret w`n' w`n'_event* w`n'_factor1 w`n'_factor7 w`n'_factor3 w`n'_factor4 w`n'_factor5 w`n'_factor6 [aw=mcap], absorb(d) vce(cluster gg d)
		forvalues i=1/7 {
			forvalues k=1/7 {
				cap replace beta`k'=_coef[w`n'_factor`k'] in `m'
				cap replace sebeta`k'=_se[w`n'_factor`k'] in `m'
				}
			lincom w`n'+w`n'_event`i'
			replace co=r(estimate) in `m'
			replace se=r(se) in `m'
			replace mult=`mdiff' in `m'
			replace ev=`i' in `m'
			replace index=`j' in `m'
			replace model=6 in `m'

			local m=`m'+1
			}
		lincom w`n'_event3*2+w`n'_event4+w`n'_event7*5-w`n'_event1*13-w`n'_event2*21-w`n'_event5*3-w`n'_event6*2	
		replace co=r(estimate) in `m'
		replace se=r(se) in `m'
		replace mult=`mdiff' in `m'
		replace ev=8 in `m'
		replace index=`j' in `m'
		replace model=6 in `m'
		local m=`m'+1
		
		local j=`j'+1
		}
		
	keep if index~=.
	keep index mult co se ev model beta* sebeta*
	save table2_results, replace
	
	/* Table 2A (Raw returns; all indexes) */
	use table2_results, clear
	drop beta* sebeta*
	keep if model==0
	rename co v1
	rename se v2
	rename mult v3
	replace v1=v1*v3
	replace v2=v2*v3
	reshape long v, i(index ev) j(var)
	replace v=v*13 if ev==1&var<3
	replace v=v*21 if ev==2&var<3
	replace v=v*2 if ev==3&var<3
	replace v=v*1 if ev==4&var<3
	replace v=v*3 if ev==5&var<3
	replace v=v*2 if ev==6&var<3
	replace v=v*5 if ev==7&var<3	
	reshape wide v, i(var ev) j(index)
	drop if var==3&ev<8
	sort ev var
	list ev var v1 v2 v3 v4 v5, sep(0)
	outsheet using table2a.csv, c replace
	
	/* Table A2 (Alphas; Overall index; all models) */
	use table2_results, clear
	drop beta* sebeta*
	keep if index==1
	rename co v1
	rename se v2
	rename mult v3
	replace v1=v1*v3
	replace v2=v2*v3
	reshape long v, i(model ev) j(var)
	replace v=v*13 if ev==1&var<3
	replace v=v*21 if ev==2&var<3
	replace v=v*2 if ev==3&var<3
	replace v=v*1 if ev==4&var<3
	replace v=v*3 if ev==5&var<3
	replace v=v*2 if ev==6&var<3
	replace v=v*5 if ev==7&var<3
	reshape wide v, i(var ev) j(model)
	drop if var==3&ev<8
	sort ev var
	list ev var v0 v1 v2 v3 v4 v5 v6, sep(0)
	outsheet using tableA2a.csv, c replace	

	
	
	use table2_temp, clear
	
	gen byte one = 1
	
	/* Window lengths:  1 = 142 days; 2 = 143 days */
	gen event = .
	replace event = 0 if d>mdy(11,8,2016)&d<=mdy(11,9,2016)		
	replace event = 1 if d>mdy(11,10,2016)&d<=mdy(6,7,2017)	
	replace event = 2 if d>mdy(6,7,2017)&d<=mdy(12,31,2017)	
	
	egen byte gg=group(ggroup)
	
	gen index=.
	gen co=.
	gen se=.
	gen mult=.
	gen ev=.
	gen model=.
	
	forvalues k=1/7 {
		gen beta`k'=.
		gen sebeta`k'=.
		}
	
	local m=1
	local j=1
	
	foreach n in _overall _pred_gsubind _resid_gsubind _pred_tax_paid _resid_tax_paid {
		quietly su ret if event==0&w`n'>0 [aw=mcap*w`n']
		local m1 = r(mean)
		quietly su ret if event==0&w`n'<0 [aw=-1*mcap*w`n']
		local m2 = r(mean)
		local mdiff = int((`m1' - `m2')*10000)/100
		
		forvalues i=1/2 {
			gen w`n'_event`i' = w`n'*(event==`i')
			}
			
		forvalues k=1/7 {
			gen w`n'_factor`k' = w`n'*factor`k'
			}
		
		/* Raw returns */
		reghdfe ret w`n' w`n'_event* [aw=mcap], absorb(d) vce(cluster gg d)
		forvalues i=1/2 {
			lincom w`n'+w`n'_event`i'
			replace co=r(estimate) in `m'
			replace se=r(se) in `m'
			replace mult=`mdiff' in `m'
			replace ev=`i' in `m'
			replace index=`j' in `m'
			replace model=0 in `m'
			local m=`m'+1
			}

		/* CAPM model */
		reghdfe ret w`n' w`n'_event* w`n'_factor1 [aw=mcap], absorb(d) vce(cluster gg d)
		forvalues i=1/2 {
			forvalues k=1/1 {
				replace beta`k'=_coef[w`n'_factor`k'] in `m'
				replace sebeta`k'=_se[w`n'_factor`k'] in `m'
				}
			lincom w`n'+w`n'_event`i'
			replace co=r(estimate) in `m'
			replace se=r(se) in `m'
			replace mult=`mdiff' in `m'
			replace ev=`i' in `m'
			replace index=`j' in `m'
			replace model=1 in `m'

			local m=`m'+1
			}			

		/* CAPM+SMB model */
		reghdfe ret w`n' w`n'_event* w`n'_factor1 w`n'_factor2 [aw=mcap], absorb(d) vce(cluster gg d)
		forvalues i=1/2 {
			forvalues k=1/2 {
				replace beta`k'=_coef[w`n'_factor`k'] in `m'
				replace sebeta`k'=_se[w`n'_factor`k'] in `m'
				}
			lincom w`n'+w`n'_event`i'
			replace co=r(estimate) in `m'
			replace se=r(se) in `m'
			replace mult=`mdiff' in `m'
			replace ev=`i' in `m'
			replace index=`j' in `m'
			replace model=2 in `m'

			local m=`m'+1
			}			
			
		/* FF3 model */
		reghdfe ret w`n' w`n'_event* w`n'_factor1 w`n'_factor2 w`n'_factor3 [aw=mcap], absorb(d) vce(cluster gg d)
		forvalues i=1/2 {
			forvalues k=1/3 {
				replace beta`k'=_coef[w`n'_factor`k'] in `m'
				replace sebeta`k'=_se[w`n'_factor`k'] in `m'
				}
			lincom w`n'+w`n'_event`i'
			replace co=r(estimate) in `m'
			replace se=r(se) in `m'
			replace mult=`mdiff' in `m'
			replace ev=`i' in `m'
			replace index=`j' in `m'
			replace model=3 in `m'

			local m=`m'+1
			}
			
		/* Carhart model */
		reghdfe ret w`n' w`n'_event* w`n'_factor1 w`n'_factor2 w`n'_factor3 w`n'_factor4 [aw=mcap], absorb(d) vce(cluster gg d)
		forvalues i=1/2 {
			forvalues k=1/4 {
				replace beta`k'=_coef[w`n'_factor`k'] in `m'
				replace sebeta`k'=_se[w`n'_factor`k'] in `m'
				}
			lincom w`n'+w`n'_event`i'
			replace co=r(estimate) in `m'
			replace se=r(se) in `m'
			replace mult=`mdiff' in `m'
			replace ev=`i' in `m'
			replace index=`j' in `m'
			replace model=4 in `m'

			local m=`m'+1
			}			

		/* FF5 model */
		reghdfe ret w`n' w`n'_event* w`n'_factor1 w`n'_factor7 w`n'_factor3 w`n'_factor5 w`n'_factor6 [aw=mcap], absorb(d) vce(cluster gg d)
		forvalues i=1/2 {
			forvalues k=1/7 {
				cap replace beta`k'=_coef[w`n'_factor`k'] in `m'
				cap replace sebeta`k'=_se[w`n'_factor`k'] in `m'
				}
			lincom w`n'+w`n'_event`i'
			replace co=r(estimate) in `m'
			replace se=r(se) in `m'
			replace mult=`mdiff' in `m'
			replace ev=`i' in `m'
			replace index=`j' in `m'
			replace model=5 in `m'

			local m=`m'+1
			}			
			
		/* FF5+UMD model */
		reghdfe ret w`n' w`n'_event* w`n'_factor1 w`n'_factor7 w`n'_factor3 w`n'_factor4 w`n'_factor5 w`n'_factor6 [aw=mcap], absorb(d) vce(cluster gg d)
		forvalues i=1/2 {
			forvalues k=1/7 {
				cap replace beta`k'=_coef[w`n'_factor`k'] in `m'
				cap replace sebeta`k'=_se[w`n'_factor`k'] in `m'
				}
			lincom w`n'+w`n'_event`i'
			replace co=r(estimate) in `m'
			replace se=r(se) in `m'
			replace mult=`mdiff' in `m'
			replace ev=`i' in `m'
			replace index=`j' in `m'
			replace model=6 in `m'

			local m=`m'+1
			}
			
		local j=`j'+1
		}
		
	keep if index~=.
	keep index mult co se ev model beta* sebeta*
	save table2b_results, replace
	
	/* Table 2B (Raw returns; all indexes) */
	use table2b_results, clear
	drop beta* sebeta*
	keep if model==0
	rename co v1
	rename se v2
	rename mult v3
	replace v1=v1*v3
	replace v2=v2*v3
	reshape long v, i(index ev) j(var)
	replace v=v*142 if ev==1&var<3
	replace v=v*143 if ev==2&var<3
	reshape wide v, i(var ev) j(index)
	drop if var==3&ev<2
	sort ev var
	list ev var v1 v2 v3 v4 v5, sep(0)
	outsheet using table2b.csv, c replace
	
	/* Table A2 (Alphas; Overall index; all models) */
	use table2b_results, clear
	drop beta* sebeta*
	keep if index==1
	rename co v1
	rename se v2
	rename mult v3
	replace v1=v1*v3
	replace v2=v2*v3	
	reshape long v, i(model ev) j(var)
	replace v=v*142 if ev==1&var<3
	replace v=v*143 if ev==2&var<3	
	reshape wide v, i(var ev) j(model)
	drop if var==3&ev<2
	sort ev var
	list ev var v0 v1 v2 v3 v4 v5 v6, sep(0)
	outsheet using tableA2b.csv, c replace	
	
	
	}	
	
if `tableA1' == 1 {
	use indices2, clear
	keep d index*_stock index*_*_gsubind index*_*_tax_paid
	
	sort d
	foreach n in _stock _pred_gsubind _resid_gsubind _pred_tax_paid _resid_tax_paid {
		foreach m in l s {
			gen ret`m'`n' = (index`m'`n')/(index`m'`n'[_n-1])-1
			}
		gen retls`n'=retl`n'-rets`n'	
		}
	
	sort d
	merge d using ffd, nokeep
	drop _merge
	
	sort d
	merge d using ff5d, nokeep
	drop _merge
	
	replace rmw=rmw/100
	replace cma=cma/100
	
	gen event = 0
	replace event = 1 if d>mdy(11,10,2016)&d<=mdy(6,7,2017)
	replace event = 2 if d>mdy(6,7,2017)&d<=mdy(12,31,2017)
	
	tab event
	
	xi i.event
	
	forvalues i = 0/6 {
		gen co`i' = .
		gen se`i' = .
		}
	
	gen port=""
	
	gen factors = .
	
	gen ev = .
	
	local n = 1
	
	foreach m in l s ls {
		reg ret`m'_stock _Ie* if event>0, robust noc
		replace factors = 0 in `n'
		replace ev=0 in `n'
		replace port="`m'" in `n'
		local n = `n' + 1
		forvalues e=1/2 {
			replace co0=_coef[_Ievent_`e'] in `n'
			replace se0=_se[_Ievent_`e'] in `n'
			replace factors = 0 in `n'
			replace ev=`e' in `n'
			replace port="`m'" in `n'
			local n = `n' + 1
			}
	
		reg ret`m'_stock mktrf _Ie* if event>0, robust noc
		replace co1=_coef[mktrf] in `n'
		replace se1=_se[mktrf] in `n'
		replace factors = 1 in `n'
		replace ev=0 in `n'
		replace port="`m'" in `n'
		local n = `n' + 1
		forvalues e=1/2 {
			replace co0=_coef[_Ievent_`e'] in `n'
			replace se0=_se[_Ievent_`e'] in `n'
			replace factors = 1 in `n'
			replace ev=`e' in `n'
			replace port="`m'" in `n'
			local n = `n' + 1
			}
		
		reg ret`m'_stock mktrf smb hml _Ie* if event>0, robust noc
		replace co1=_coef[mktrf] in `n'
		replace se1=_se[mktrf] in `n'		
		replace co2=_coef[smb] in `n'
		replace se2=_se[smb] in `n'
		replace co3=_coef[hml] in `n'
		replace se3=_se[hml] in `n'		
		replace factors = 3 in `n'
		replace ev=0 in `n'
		replace port="`m'" in `n'
		local n = `n' + 1
		forvalues e=1/2 {
			replace co0=_coef[_Ievent_`e'] in `n'
			replace se0=_se[_Ievent_`e'] in `n'
			replace factors = 3 in `n'
			replace ev=`e' in `n'
			replace port="`m'" in `n'
			local n = `n' + 1
			}
		
		
		reg ret`m'_stock mktrf smb hml umd _Ie* if event>0, robust noc
		replace co1=_coef[mktrf] in `n'
		replace se1=_se[mktrf] in `n'		
		replace co2=_coef[smb] in `n'
		replace se2=_se[smb] in `n'
		replace co3=_coef[hml] in `n'
		replace se3=_se[hml] in `n'		
		replace co4=_coef[umd] in `n'
		replace se4=_se[umd] in `n'
		replace factors = 4 in `n'
		replace ev=0 in `n'
		replace port="`m'" in `n'
		local n = `n' + 1
		forvalues e=1/2 {
			replace co0=_coef[_Ievent_`e'] in `n'
			replace se0=_se[_Ievent_`e'] in `n'
			replace factors = 4 in `n'
			replace ev=`e' in `n'
			replace port="`m'" in `n'
			local n = `n' + 1
			}
		
		reg ret`m'_stock mktrf smb hml rmw cma _Ie* if event>0, robust noc
		replace co1=_coef[mktrf] in `n'
		replace se1=_se[mktrf] in `n'		
		replace co2=_coef[smb] in `n'
		replace se2=_se[smb] in `n'
		replace co3=_coef[hml] in `n'
		replace se3=_se[hml] in `n'		
		replace co5=_coef[rmw] in `n'
		replace se5=_se[rmw] in `n'
		replace co6=_coef[cma] in `n'
		replace se6=_se[cma] in `n'		
		replace factors = 5 in `n'
		replace ev=0 in `n'
		replace port="`m'" in `n'
		local n = `n' + 1		
		forvalues e=1/2 {
			replace co0=_coef[_Ievent_`e'] in `n'
			replace se0=_se[_Ievent_`e'] in `n'
			replace factors = 5 in `n'
			replace ev=`e' in `n'
			replace port="`m'" in `n'
			local n = `n' + 1
			}
		
		reg ret`m'_stock mktrf smb hml umd rmw cma _Ie* if event>0, robust noc
		replace co1=_coef[mktrf] in `n'
		replace se1=_se[mktrf] in `n'		
		replace co2=_coef[smb] in `n'
		replace se2=_se[smb] in `n'
		replace co3=_coef[hml] in `n'
		replace se3=_se[hml] in `n'		
		replace co4=_coef[umd] in `n'
		replace se4=_se[umd] in `n'
		replace co5=_coef[rmw] in `n'
		replace se5=_se[rmw] in `n'
		replace co6=_coef[cma] in `n'
		replace se6=_se[cma] in `n'		
		replace factors = 6 in `n'
		replace ev=0 in `n'
		replace port="`m'" in `n'
		local n = `n' + 1
		forvalues e=1/2 {
			replace co0=_coef[_Ievent_`e'] in `n'
			replace se0=_se[_Ievent_`e'] in `n'
			replace factors = 6 in `n'
			replace ev=`e' in `n'
			replace port="`m'" in `n'
			local n = `n' + 1
			}
		}
		
		keep co* se* port f factors ev
		keep if port~=""
		reshape long co se, i(factors ev port) j(f)
		rename co v1
		rename se v2
		reshape long v, i(factors ev port f) j(var)
		keep if v~=.
		
		replace port="z" if port=="ls"
		
		keep if port=="z"&ev==0
		reshape wide v, i(factors var) j(f)
		list factors var v*, sep(0)
		sort factors var
		outsheet factors v* using tableA1.csv, replace c		
		
	}	

/* Graphs */	
if `graphs' == 1 {
	set scheme s1color

	/* Figure 1 graph */

	use indices2, clear
	keep d ls_stock ls_*_gsubind ls_*_tax_*
	label var d "Date"
	label var ls_stock "Stock-level index"
	label var ls_pred_gsubind "Between-industry index"
	label var ls_resid_gsubind "Within-industry index"
	label var ls_pred_tax_paid "Correlated with cash tax rate"
	label var ls_resid_tax_paid "Orthogonal to cash tax rate"
	label var ls_pred_tax_total "Correlated with cash tax rate"
	label var ls_resid_tax_total "Orthogonal to cash tax rate"

	foreach v of varlist ls_stock ls_*_gsubind ls_*_tax_* {
		replace `v'=`v'+100
		}
		
	keep if d>=mdy(11,8,2016)
	keep if year(d)<2018

	* S&P and prediction market
	gen rup1=116 if d>=mdy(3,7,2017)&d<=mdy(3,24,2017)
	gen rdn1=95 if d>=mdy(3,7,2017)&d<=mdy(3,24,2017)

	gen rup2=116 if d>=mdy(5,8,2017)&d<=mdy(6,9,2017)
	gen rdn2=95 if d>=mdy(5,8,2017)&d<=mdy(6,9,2017)

	gen rup3=116 if d>=mdy(7,24,2017)&d<=mdy(7,28,2017)
	gen rdn3=95 if d>=mdy(7,24,2017)&d<=mdy(7,28,2017)

	gen rup4=116 if d>=mdy(10,26,2017)&d<=mdy(10,30,2017)
	gen rdn4=95 if d>=mdy(10,26,2017)&d<=mdy(10,30,2017)

	gen rup5=116 if d>=mdy(11,27,2017)&d<=mdy(12,4,2017)
	gen rdn5=95 if d>=mdy(11,27,2017)&d<=mdy(12,4,2017)

	label var rup1 "AHCA considered"
	label var rup2 "Comey fired/testifies"
	label var rup3 "ACA repeal considered"
	label var rup4 "Manafort indictment"
	label var rup5 "Tax cuts pass Senate"

	twoway ///
		(rarea rup1 rdn1 d, sort fcolor(gs13) lcolor(gs13)) ///
		(rarea rup2 rdn2 d, sort fcolor(gs13) lcolor(gs13)) ///
		(rarea rup3 rdn3 d, sort fcolor(gs13) lcolor(gs13)) ///
		(rarea rup4 rdn4 d, sort fcolor(gs13) lcolor(gs13)) ///
		(rarea rup5 rdn5 d, sort fcolor(gs13) lcolor(gs13)) ///
		(line ls_stock d, sort color(blue green)) ///
		, ///
			xlabel( ///
				`=mdy(12,1,2016)' "Dec" ///
				`=mdy(2,1,2017)' "Feb" ///
				`=mdy(4,1,2017)' "Apr" ///
				`=mdy(6,1,2017)' "Jun" ///
				`=mdy(8,1,2017)' "Aug" ///
				`=mdy(10,1,2017)' "Oct" ///
				`=mdy(12,1,2017)' "Dec" ///			
			) 		///
			ylabel(95 "95" 100 "100" 105 "105" 110 "110" 115 "115" 120 "120", ///
				labcolor(blue) angle(horizontal)) ///
			legend(off) ///
			text(118.5 `=mdy(2,10,2017)' "AHCA in House", placement(1) color(black) size(small)) ///
			text(117 `=mdy(5,7,2017)' "Comey", placement(1) color(black) size(small)) ///
			text(118.5 `=mdy(6,15,2017)' "ACA in Senate", placement(1) color(black) size(small)) ///
			text(117 `=mdy(9,15,2017)' "Manafort indicted", placement(1) color(black) size(small)) ///
			text(118.5 `=mdy(11,10,2017)' "Tax cuts", placement(1) color(black) size(small)) ///		
			text(117 `=mdy(11,15,2016)' "Trump index", placement(1) color(blue) size(small)) ///
			ytick(, notick) ///
			ytitle("Index (11/8/16 = 100)", color(blue)) ///
			xtitle(" " "2016-17") ///
			xsize(10) ysize(7.5) name(fig1, replace)

	graph save "Fig1", replace
	graph export "Fig1.png", replace
	graph export "Fig1.eps", replace


	twoway ///
		(rarea rup1 rdn1 d, sort fcolor(gs13) lcolor(gs13)) ///
		(rarea rup2 rdn2 d, sort fcolor(gs13) lcolor(gs13)) ///
		(rarea rup3 rdn3 d, sort fcolor(gs13) lcolor(gs13)) ///
		(rarea rup4 rdn4 d, sort fcolor(gs13) lcolor(gs13)) ///
		(rarea rup5 rdn5 d, sort fcolor(gs13) lcolor(gs13)) ///
		(line ls_pred_gsubind ls_resid_gsubind d, sort color(blue green) lpattern(solid dash)) ///
		, ///
			xlabel( ///
				`=mdy(12,1,2016)' "Dec" ///
				`=mdy(2,1,2017)' "Feb" ///
				`=mdy(4,1,2017)' "Apr" ///
				`=mdy(6,1,2017)' "Jun" ///
				`=mdy(8,1,2017)' "Aug" ///
				`=mdy(10,1,2017)' "Oct" ///
				`=mdy(12,1,2017)' "Dec" ///			
			) 		///
			ylabel(95 "95" 100 "100" 105 "105" 110 "110" 115 "115" 120 "120", ///
			labcolor(blue) angle(horizontal)) ///
			legend(off) ///
			text(118.5 `=mdy(2,10,2017)' "AHCA in House", placement(1) color(black) size(small)) ///
			text(117 `=mdy(5,7,2017)' "Comey", placement(1) color(black) size(small)) ///
			text(118.5 `=mdy(6,15,2017)' "ACA in Senate", placement(1) color(black) size(small)) ///
			text(117 `=mdy(9,15,2017)' "Manafort indicted", placement(1) color(black) size(small)) ///
			text(118.5 `=mdy(11,10,2017)' "Tax cuts", placement(1) color(black) size(small)) ///		
			text(114 `=mdy(11,15,2016)' "Between-industry (solid)", placement(1) color(blue) size(small)) ///
			text(106 `=mdy(11,15,2016)' "Within-industry (dashed)", placement(1) color(green) size(small)) ///		
			ytick(, notick) ///
			ytitle("Index (11/8/16 = 100)", color(blue)) ///
			xtitle(" " "2016-17") ///
			graphregion(color(white)) bgcolor(white) xsize(10) ysize(7.5) name(fig2A, replace)

	graph save "Fig2A", replace
	graph export "Fig2A.png", replace
	graph export "Fig2A.eps", replace

	twoway ///
		(rarea rup1 rdn1 d, sort fcolor(gs13) lcolor(gs13)) ///
		(rarea rup2 rdn2 d, sort fcolor(gs13) lcolor(gs13)) ///
		(rarea rup3 rdn3 d, sort fcolor(gs13) lcolor(gs13)) ///
		(rarea rup4 rdn4 d, sort fcolor(gs13) lcolor(gs13)) ///
		(rarea rup5 rdn5 d, sort fcolor(gs13) lcolor(gs13)) ///
		(line ls_pred_tax_paid ls_resid_tax_paid d, sort color(blue green) lpattern(solid dash)) ///
		, ///
			xlabel( ///
				`=mdy(12,1,2016)' "Dec" ///
				`=mdy(2,1,2017)' "Feb" ///
				`=mdy(4,1,2017)' "Apr" ///
				`=mdy(6,1,2017)' "Jun" ///
				`=mdy(8,1,2017)' "Aug" ///
				`=mdy(10,1,2017)' "Oct" ///
				`=mdy(12,1,2017)' "Dec" ///			
			) 		///
			ylabel(95 "95" 100 "100" 105 "105" 110 "110" 115 "115" 120 "120", ///
			labcolor(blue) angle(horizontal)) ///
			legend(off) ///
			text(118.5 `=mdy(2,10,2017)' "AHCA in House", placement(1) color(black) size(small)) ///
			text(117 `=mdy(5,7,2017)' "Comey", placement(1) color(black) size(small)) ///
			text(118.5 `=mdy(6,15,2017)' "ACA in Senate", placement(1) color(black) size(small)) ///
			text(117 `=mdy(9,15,2017)' "Manafort indicted", placement(1) color(black) size(small)) ///
			text(118.5 `=mdy(11,10,2017)' "Tax cuts", placement(1) color(black) size(small)) ///		
			text(106 `=mdy(11,15,2016)' "Correlated with taxes (solid)", placement(1) color(blue) size(small)) ///
			text(116 `=mdy(11,15,2016)' "Orthogonal to taxes (dashed)", placement(1) color(green) size(small)) ///		
			ytick(, notick) ///
			ytitle("Index (11/8/16 = 100)", color(blue)) ///
			xtitle(" " "2016-17") ///
			xsize(10) ysize(7.5) name(fig2B, replace)

	graph save "Fig2B", replace
	graph export "Fig2B.png", replace
	graph export "Fig2B.eps", replace


	use trump_departure, clear
	keep dt betfair2*
	gen survive2017 = 1-betfair2017
	gen survive2018 = 1-betfair2017-betfair2018
	gen survive2019 = 1-betfair2017-betfair2018-betfair2019

	gen d=dofc(dt)
	sort d 
	merge d using reelect
	drop _merge

	rename trump_reelect reelect

	sort d
	merge d using approval
	drop _merge

	replace a1=a1/100

	keep if d>=mdy(1,20,2017)

	gen rup1=.75 if d>=mdy(3,7,2017)&d<=mdy(3,24,2017)
	gen rdn1=0.1 if d>=mdy(3,7,2017)&d<=mdy(3,24,2017)

	gen rup2=.75 if d>=mdy(5,8,2017)&d<=mdy(6,9,2017)
	gen rdn2=0.1 if d>=mdy(5,8,2017)&d<=mdy(6,9,2017)

	gen rup3=.75 if d>=mdy(7,24,2017)&d<=mdy(7,28,2017)
	gen rdn3=0.1 if d>=mdy(7,24,2017)&d<=mdy(7,28,2017)

	gen rup4=.75 if d>=mdy(10,26,2017)&d<=mdy(10,30,2017)
	gen rdn4=0.1 if d>=mdy(10,26,2017)&d<=mdy(10,30,2017)

	gen rup5=.75 if d>=mdy(11,27,2017)&d<=mdy(12,4,2017)
	gen rdn5=0.1 if d>=mdy(11,27,2017)&d<=mdy(12,4,2017)

	twoway ///
		(rarea rup1 rdn1 dt, sort fcolor(gs13) lcolor(gs13)) ///
		(rarea rup2 rdn2 dt, sort fcolor(gs13) lcolor(gs13)) ///
		(rarea rup3 rdn3 dt, sort fcolor(gs13) lcolor(gs13)) ///
		(rarea rup4 rdn4 dt, sort fcolor(gs13) lcolor(gs13)) ///
		(rarea rup5 rdn5 dt, sort fcolor(gs13) lcolor(gs13)) ///
		(line survive2019 reelect a1 dt, sort color(blue green purple) lpattern(solid dash shortdash)) ///
		, ///
			xlabel( ///
				`=mdyhms(2,1,2017,0,0,0)' "Feb" ///
				`=mdyhms(4,1,2017,0,0,0)' "Apr" ///
				`=mdyhms(6,1,2017,0,0,0)' "Jun" ///
				`=mdyhms(8,1,2017,0,0,0)' "Aug" ///
				`=mdyhms(10,1,2017,0,0,0)' "Oct" ///
				`=mdyhms(12,1,2017,0,0,0)' "Dec" ///			
			) 		///
			ylabel(0 "0" .25 "25%" .5 "50%" .75 "75%", ///
				labcolor(blue) angle(horizontal)) ///
			legend(off) ///		
			ytick(, notick) ///
			ytitle("Probability that Trump", color(blue)) ///
			xtitle(" " "2016-17") ///
			text(0.05 `=mdyhms(2,10,2017,0,0,0)' "ACA in House", placement(1) color(black) size(small)) ///
			text(0.05 `=mdyhms(5,9,2017,0,0,0)' "Comey", placement(1) color(black) size(small)) ///
			text(0.05 `=mdyhms(6,23,2017,0,0,0)' "ACA in Senate", placement(1) color(black) size(small)) ///
			text(0.01 `=mdyhms(9,18,2017,0,0,0)' "Manafort indicted", placement(1) color(black) size(small)) ///
			text(0.05 `=mdyhms(11,12,2017,0,0,0)' "Tax cuts", placement(1) color(black) size(small)) ///		
			text(0.56 `=mdyhms(8,1,2017,0,0,0)' "Survives 2019", placement(1) color(blue) size(small)) ///
			text(0.21 `=mdyhms(8,1,2017,0,0,0)' "Is Reelected", placement(1) color(green) size(small)) ///
			text(0.33 `=mdyhms(8,1,2017,0,0,0)' "Job approval (adults)", placement(1) color(purple) size(small)) ///	
			xsize(10) ysize(7.5) name(figA1, replace)

	graph save "FigA1", replace
	graph export "FigA1.png", replace
	graph export "FigA1.eps", replace
	}
