% This file is for solving/estimating the New Keynesian model. It requires
% the following files (see ReadMe.pdf for descriptions):
%
%   - MFNK_Est_fn.m
%   - MFNK_Est_FrmData.m
%   - MFNK_Est_prtr.m
%   - MFNK_Est_SimMom.m
%   - TaylExp.m
%   - NKRF_SlvMod_Full.m
%   - NKRF_Sim_fn_mex.mexw64 (see ReadMe.pdf)
%   - DervFll_ord2_fn.m
%   - DervFll_ord3_fn.m
%   - NKRF_Derivs_Full.m (optional; see ReadMe.pdf)
%   - rgb.m
%   - bpfilter.m
%   - EstimationData.mat
%   - EstShkSd.mat
%
% For Beaudry, Galizia, and Portier, Putting the Cycle Back into Business
% Cycle Analysis (2019).

%% Initialize

clearvars
format shortG

%% Model options

ord = 3;            % perturbation order (don't change)
rediff = 0;         % 1 = re-obtain system solution functions (shouldn't need to)

ctofffct = 10;      % factor to multiply max value of hours in data by to give explosion cutoff

sedisp = 1;         % 1 = display s.e.'s
secmp = 0;          % 1 = re-compute s.e.'s; 0 = use saved values
nse = 200;          % # of simulated data sets for s.e. estimation
tlse = 1e-2;        % tolerance on s.e. bootstrap estimations
CIpct = 90;         % CI size

% Each row of the cell array cmms corresponds to a different higher moment
% in l (hours) and rp (risk prem.). In particular, row j of mms is for
% moment(j), where
%
%                E[l^mms{j,1}*rp^mms{j,2}]
% moment(j) = --------------------------------
%              sd(l)^mms{j,1}*sd(rp)^mms{j,2}
%
% The third column of cmms is the latex name of the moment (for display).
%
%       l,      rp,       name
cmms = {1,      1,      'corr$(l,r^p)$' ;...
        3,      0,      'skew$(l)$'     ;...
        0,      3,      'skew$(r^p)$'   ;...
        4,      0,      'kurt$(l)$'     ;...
        0,      4,      'kurt$(r^p)$'   ;...
        };

nsim = 3000;        % number of datasets to simulate to get model spectrum
Tbrn = 100;         % number of periods to burn in simulations

prmn = 2;           % lowest periodicity to consider in estimation
prmx = 50;          % highest periodicity to consider in estimation

frmn = 1/60;        % lowest frequency to plot
frmx = 1/4;         % highest frequency to plot


%% Specify parameters to be estimated and restrictions on them

% EstPars is a cell variable with the names of the estimated parameters.
% Second element of each row gives the restriction on variable to be
% imposed during estimation. In the following table, for each type of
% restrction, x represents an unrestricted auxiliary parameter defined on
% (-infty,infty), while y represents the structural parameter that should
% be restricted. The second column of the table shows the mapping from x to 
% y that operationalizes the restriction, while the third column shows the
% inverse mapping that obtains x from y.
%       RESTRICTION             x -> y MAPPING          y -> x MAPPING
%       0 = no restriction      x -> x,                 y -> y
%       1 = between -1 and 1    x -> (e^x-1)/(1+e^x),	y -> log((1+y)/(1-y))
%       2 = less than 1         x -> 1-e^x,             y -> log(1-y)
%       3 = greater than 1      x -> 1+e^x,             y -> log(y-1)
%       4 = less than 0         x -> -exp(x),           y -> log(-y)
%       5 = between 0 and 1     x -> e^x/(1+e^x)        y -> log(y/(1-y))
%       6 = >=0                 x -> x^2,               y -> sqrt(y)

%           PARAM,      RESTRICTION
EstPars = {                         ...
            'om',       6           ;...    risk aversion param.
            'gam_',     5           ;...    used to get gamma (habit)
            'psi_',     5           ;...    used to get psi (1-initial depr.)
            'phi_e_',   6           ;...    used to get phi_e (Taylor rule)
            'phi',      5           ;...    Pr{loan recovery}
            'Phi_',     5           ;...    used to get Phi (loan recovery cost)
            'r2til',    0           ;...    used to get vrho2 (2nd-order risk prem. coeff.)
            'r3til_',   6           ;...    used to get vrho3 (3rd-order risk prem. coeff.)
            'rho',      1           ;...    autocorrelation of shock
            'sight',    6           ;...    s.d. of shock innovation
            };
        
%% Optimization options

reopt = 0;      % 1 = re-optimize; 0 = use saved parameter values
svdshk = 1;     % 1 = load saved shock values; 0 = generate new values (CAUTION:
                    % will overwrite existing file containing seed)
init = 1;       % 1 = initialize with saved parameter values; 0 = use manual ones
vrb = 0;        % 1 = verbose output; set to 0 for estimation

% suffix of file name for initial parameters (only used if init==1); see
% ReadMe.pdf for description of files containing parameter estimates in
% paper.
fnmsuf = '2_50';

% parfnm = 'MFNK_lhnfbBLS_2_50_Canon';
% parfnmin = 'MFNK_lhnfbBLS_2_50_NoFriction';
% parfnm = 'MFNK_lhnfbBLS_2_50_LinRP';
% parfnmin = 'MFNK_lhnfbBLS_2_50';
% parfnmin = 'ParBackup';    

% fminsearch options
mnopt.fminsopt = optimset('display','iter','MaxFunEvals',2000,'MaxIter',2000,...
    'TolFun',1e-3,'TolX',1e-3);
mnopt.nruns = 8;     % number of times to re-optimize

%% Fixed model parameters and manually specified initial ones

% Fixed parameters
del = 0.05;     % depreciation rate
al = .67;       % elasticity of output w.r.t. labor
e = 1-0.0583;   % steady state employment rate

% Manual initial parameters (only used if init==0). Parameters should be
% loaded into struct variable MP, which will be passed to other functions.
MP.om = 0.299750569633800;
MP.gam = 0.533460126364182;
MP.psi = 0.399954383918036;
MP.phi_e = 0.0421084035450442;
MP.phi = 0.866777533219643;
MP.Phi = 0.0421337159186503;
MP.r2 = 0.0334741476709093;     % 2 x 2nd-order risk-premium coeff. (vrho2)
MP.r3 = 3.55713601077074;       % 6 x 3rd-order risk-premium coeff. (vrho3)
MP.rho = -3.42641333691840e-07;
MP.sight = 0.000156522462617143; 

%% Plotting options

plfig = 0;                  % 1 = plot figures
fign = 1;                   % figure number to use
fsz = 16;                   % font size for axis/legend
tfsz = 16;                  % font size for title
lnwd = 4;                   % line width for plots
figpos = .8*[0 0 1058 817]; % figure position

%% Data options

flnm = 'EstimationData';      % file name with data in it

% Date range to use for hours
frstl = 1948;           % first observation
lstl = 2015.25;         % last observation

% Date range to use for risk premium
frstRP = 1954.5;       % first observation
lstRP = 2015.25;       % last observation


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


%% Read in data and detrend

load(flnm)      % load file containing data (also contains dates of
                % observations in the variable ddate)

smpl = (ddate >= frstl) & (ddate <= lstl);      % indices of hours observations used from full sample
smpRP = (ddate >= frstRP) & (ddate <= lstRP);	% indices of risk premium observations used from full sample


ddatel = ddate(smpl);       % vector of dates used for hours
ddateRP = ddate(smpRP);     % vector of dates used in computing trend

comml = (ddatel >= frstRP) & (ddatel <= lstRP);     % indices of hours subsample for dates also in risk premium subsample
commRP = (ddateRP >= frstl) & (ddateRP <= lstl);    % indices of risk premium subsample for dates also in hours subsample

Tl = nnz(smpl);             % number of hours observations used
TRP = nnz(smpRP);           % number of risk premium observations used
Tsim = max(Tl,TRP);         % total number of periods spanned by subsample

% convert hours series to percentage, drop unused observations, and load into variable l
l = 100*lhnfbBLS(smpl);	 
l = l-mean(l);                  % de-mean

lbp = bpfilter(l,prmn,prmx);    % band-pass filter (for moments)
lbp = lbp-mean(lbp);            % de-mean
lbp = lbp(comml);               % drop non-common observations

% drop unused risk premium observations and load into variable RP
RP = Spread1(smpRP);
RP = RP-mean(RP);               % de-mean

RPbp = bpfilter(RP,prmn,prmx);  % band-pass filter (for moments)
RPbp = RPbp-mean(RPbp);         % de-mean
RPbp = RPbp(commRP);            % drop non-common observations



%% Draw shock innovation values

Ttot = 4*(nsim*Tsim + Tbrn);    % total periods to simulate (four times as many as minimum)

if svdshk == 1      % if using saved random number generator seed
    if ~exist('EstShkSd.mat', 'file')       % make sure file containing seed exists
        error('Saved shock seed not found. Set svdshk = 0 to generate new seed.')
    end
    load EstShkSd                           % load file
else                % otherwise generate a new seed and overwrite old one (if it exists)
    rng('shuffle')
    scurr=rng;
    shksd = scurr.Seed;
    save EstShkSd shksd
end

rng(shksd)              % seed random number generator
shk = randn(1,Ttot);    % draw shocks

%% Set spectrum parameters and get data spectra

TT = 1024;                  % number of points in Fourier transform

% this block computes the number of points in one side of the Fourier
% transform
Todd = logical(mod(TT,2));  % flag=true if TT is odd
if Todd
    TT2 = (TT+1)/2;
    freqs = linspace(0,.5*(TT-1)/TT,TT2);      % vector of ordinary frequencies, evenly spaced
else
    TT2 = TT/2 + 1;
    freqs = linspace(0,.5,TT2)';        % vector of ordinary frequencies, evenly spaced
end

nkr = 13;           % length of kernel for smoothing
kr = hamming(nkr);  % kernel
kr = kr/sum(kr);	% normalize weights to sum to one

% get smoothed hours spectrum
sl2 = abs(fft(l,TT)).^2;        % un-scaled two-sided spectrum
sl1 = sl2(1:TT2)/(2*pi*Tl);     % scaled one-sided spectrum
conspecl = [sl1(nkr:-1:2);sl1;sl1(end-1:-1:end-nkr+1)]; % extend for wrap-around kernel-smoothing
conspecl = conv(conspecl,kr,'same');        % kernel-smooth
datspecl = conspecl(nkr:TT2+nkr-1);         % drop extra frequencies added for wrap-around

% smoothed risk premium spectrum
sRP2 = abs(fft(RP,TT)).^2;          % un-scaled two-sided spectrum
sRP1 = sRP2(1:TT2)/(2*pi*TRP);      % scaled one-sided spectrum
conspecRP = [sRP1(nkr:-1:2);sRP1;sRP1(end-1:-1:end-nkr+1)]; % extend for wrap-around kernel-smoothing
conspecRP = conv(conspecRP,kr,'same');      % kernel-smooth
datspecRP = conspecRP(nkr:TT2+nkr-1);       % drop extra frequencies added for wrap-around

spcmp = (freqs>=1/prmx) & (freqs<=1/prmn);  % indices of frequencies to target in estimation
spcpl = (freqs>=frmn) & (freqs<=frmx);      % indices of frequencies to plot


%% BP-filtering matrices

% These matrices, when pre-multiplying the correct size data series, will
% BP-filter it.
[~,BPmatl] = bpfilter(randn(Tl,1),prmn,prmx,1);
[~,BPmatRP] = bpfilter(randn(TRP,1),prmn,prmx,1);

%% Higher moments

mms = cell2mat(cmms(:,1:2));    % extract exponents for higher moments
mmnm = cmms(:,3);               % labels for moments
nmm = size(mms,1);              % number of moments

sigl = std(lbp,1);                  % s.d. of BP-filtered hours
sigRP = std(RPbp,1);                % s.d. of BP-filtered risk premium
den = prod([sigl,sigRP].^mms,2);    % vector of denominators for moments
mmsrsh = shiftdim(mms',-1);         % convert mms to 1-by-2-by-nmm array where mmsrsh(:,:,j)
                                        % is the j-th row of mms
% compute higher moments
datmm = reshape(mean(lbp.^mmsrsh(:,1,:).*RPbp.^mmsrsh(:,2,:)),...
    [1,nmm])'./den;


%% Load initial model parameters from file if necessary

if init == 1                % if initializing from saved parameter values
    parfnm = ['Params_' fnmsuf];
    load(parfnm)            % load file containing previous parameter estimates;
                                % parameters should be contained in a
                                % struct variable called MPsv
    MP = MPsv;              % copy saved parameters to struct variable MP
end

%% Set up struct variable MP containing all model parameters

MP.del = del;
MP.al = al;
MP.e = e;

% derived parameters (including auxiliary ones for estimation)
MP.Xi = (1-MP.phi)/(1+(1-MP.e)*MP.phi/MP.e);
MP.tau = (1-MP.gam)*(MP.del+MP.psi)/MP.om;
MP.kap = MP.al*MP.del*(1+MP.gam-MP.psi)+MP.tau*MP.Xi;
MP.Rp = (1+((1-MP.e)*(1-MP.phi))*MP.Phi_)/(MP.e+(1-MP.e)*MP.phi);
if MP.phi == 1
    MP.Phi_ = 0;
else
    MP.Phi_ = MP.Phi*MP.phi/(1-MP.phi);
end
MP.r1 = -(1-MP.phi+MP.phi*MP.Phi)*MP.e/(MP.e+(1-MP.e)*MP.phi)^2;
if MP.r1 >= 1
    error('Complementarity too strong (multiple equilibria).')
end
MP.r2til = MP.r2/(1-MP.phi);
MP.r3til = MP.r3/(1-MP.phi);
MP.r3til_ = MP.r3til - (1-MP.phi)*MP.r2til^2/(2*(MP.kap/MP.tau+MP.r1));
if MP.r3til_ <0 
    error('Third-order coefficient should be non-negative.')
end
MP.phi_e_ = MP.phi_e*MP.tau/(MP.al*MP.del);
if MP.phi_e_ >= 1
    error('Forward-looking term enters negatively')
end
MP.psi_ = MP.psi/((1-MP.del)*.99);
MP.gam_ = MP.gam/((1-MP.del)*.99);

%% Transform initial parameters to embody parameter restrictions (see section
 % above on specifying parameters for estimation)

nprEst = size(EstPars,1);   % number of estimated parameters
parsin = zeros(nprEst,1);   % vector for transformed initial parameters

for j = 1:nprEst            % for each estimated parameter
    prNT = MP.(EstPars{j,1});   % untransformed initial paramater value
    switch EstPars{j,2}         % depending on variable indicating restriction type,
                                    % do corresponding transformation
                                    
        case 0                      % no restriction case
            prT = prNT;
        case 1                      % between -1 and 1
            prT = log((1+prNT)/(1-prNT));
        case 2                      % less than 1
            prT = log(1-prNT);
        case 3                      % greater than 1
            prT = log(prNT-1);
        case 4                      % less than 0
            prT = log(-prNT);
        case 5                      % between 0 and 1
            prT = log(prNT/(1-prNT));
        case 6                      % non-negative
            prT = sqrt(prNT);
    end
    parsin(j) = prT;            % load into parsin vector
end

%% Other parameters

% for solution
nstslv = 4;
[expnsslv,cmbsslv] = TaylExp(nstslv,ord);	% see TaylExp.m for explanation
ntrmslv = size(expnsslv,1);                 % number of monomials in polynomial

% for simulation (drop perturbation parameter as state variable)
nstsim = 3;
[expnssim,~] = TaylExp(nstsim,ord);         % see TaylExp.m for explanation
expnssim = [zeros(1,3);expnssim];           % add row for constant term
ntrmsim = size(expnssim,1);                 % number of monomials in polynomial

if rediff               % if re-obtaining derivative functions
    NKRF_Derivs_Full(ord,expnsslv,cmbsslv)  % call function to do so
end

% Initial state for simulations (was randomly generated). Note: First Tbrn
% observations of simulations will be discarded, so effect of initial
% choice should be minimal.
St0 = [-2.48755125295155;1.20920228958175;0.237498959476462]/100;

% parameters controlling when a simulation will be deemed to have exploded
mxl = max(abs(l));          % largest value of de-meaned log-hours in absolute value
ctoff = ctofffct*mxl;       % multiply this value by ctofffct to get maximum allowed
                                % value of |log-hours| in a simulation

%% Load necessary non-model variables into struct variable OP (will be passed to other
 % functions).
 
OP.smpl = smpl;     OP.smpRP = smpRP;	OP.Tl = Tl;         OP.TRP = TRP;
OP.Tsim = Tsim;     OP.comml = comml;   OP.commRP = commRP; OP.St0 = St0;
OP.Ttot = Ttot;     OP.Tbrn = Tbrn;     OP.nsim = nsim;     OP.TT = TT;
OP.TT2 = TT2;       OP.kr = kr;         OP.nkr = nkr;       OP.spcmp = spcmp;
OP.spcpl = spcpl;   OP.BPmatl = BPmatl; OP.BPmatRP = BPmatRP;   OP.mms = mms;
OP.nmm = nmm;       OP.mmsrsh = mmsrsh; OP.nprEst = nprEst; OP.ord = ord;
OP.expnsslv = expnsslv; OP.cmbsslv = cmbsslv;   OP.ntrmslv = ntrmslv;
OP.expnssim = expnssim; OP.ntrmsim = ntrmsim;   OP.vrb = vrb;
OP.ctoff = ctoff;

%% Load data moments to be matched into struct variable MMdat (will be passed to other
 % functions.
 
MMdat.specl = datspecl;
MMdat.specRP = datspecRP;
MMdat.datmm = datmm;

%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Estimation (if necessary)

if reopt        % if re-estimating
    parsopt = MFNK_Est_FrmData(MMdat,shk,MP,OP,EstPars,parsin,mnopt);    % call function to do so
else            % otherwise
    parsopt = parsin;	% set "optimum" parameters to initial ones
end


%% Extract optimal parameters and solve model at optimum

% call function to convert transformed parameters back to un-transformed
% ones
MP = MFNK_Est_prtr(parsopt,MP,EstPars); 

% recover objective function value (fnvl), coefficients for solution
% (Copt), model moments (MM), and full model simulation results (SM)
[fnvl,coefs,MM,SM] = MFNK_Est_fn(parsopt,MMdat,shk,MP,OP,EstPars);

% reduced-form parameters
prs = [MP.del MP.psi MP.xi MP.al2 MP.al1 MP.rho];
Dth = [0; MP.Dth2; MP.Dth3];
prsall = [prs Dth'];

% get eigenvalues of solved system and sort by modulus
[~,~,eigs] = NKRF_SlvMod_Full(prsall,OP);
[~,srteig] = sort(eigs(:,3));
eigs = eigs(srteig,:);

%% Standard errors (if necessary)

if sedisp == 1      % if displaying standard errors
    if secmp == 1       % if re-computing standard errors
        
        % check to make sure we obtained enough simulated data sets
        if nse > SM.nact
            error('Not enough simulated data sets in SM.')
        end
        
        % extract nse data sets and smooth the associated spectra
        speclall = SM.speclall(:,1:nse);
        conspecl = [speclall(nkr:-1:2,:);speclall;speclall(end-1:-1:end-nkr+1,:)]; % extend for wrap-around kernel-smoothing
        conspecl = conv2(conspecl,kr,'same');       % kernel-smooth
        speclall = conspecl(nkr:TT2+nkr-1,:);     % drop extra frequencies added for wrap-around
        specRPall = SM.specRPall(:,1:nse);
        conspecRP = [specRPall(nkr:-1:2,:);specRPall;specRPall(end-1:-1:end-OP.nkr+1,:)]; % extend for wrap-around kernel-smoothing
        conspecRP = conv2(conspecRP,kr,'same');       % kernel-smooth
        specRPall = conspecRP(nkr:TT2+nkr-1,:);     % drop extra frequencies added for wrap-around
        mmsall = SM.mmsall(:,1:nse);
        
        parsoptsim = zeros(nprEst,nse);     % matrix to hold optimum transformed parameters
                                                % from estimations on simulated data sets
        psim = zeros(11,nse);               % matrix to hold results for displayed parameters
        
        % fminsearch options
        mnoptse.fminsopt = optimset('display','off','MaxFunEvals',2000,'MaxIter',2000,...
            'TolFun',tlse,'TolX',tlse);
        mnoptse.nruns = 1;     % number of times to re-optimize
        
        for jse = 1:nse         % for each simulated data set
            
            % get "data" moments
            MMj.specl = speclall(:,jse);
            MMj.specRP = specRPall(:,jse);
            MMj.datmm = mmsall(:,jse);
            
            % estimate parameters using "data" moments
            parsinj = parsopt;
            parsoptj = MFNK_Est_FrmData(MMj,shk,MP,OP,EstPars,parsinj,mnoptse);
            
            parsoptsim(:,jse) = parsoptj;               % retain optimized transformed parameters
            
            MPj = MFNK_Est_prtr(parsoptj,MP,EstPars);   % extract un-transformed parameters
            
            % retain optimized displayed parameters
            psim(:,jse) = [             ...
                MPj.om      ;...
                MPj.gam     ;...
                MPj.psi     ;...
                MPj.phi_e	;...
                MPj.phi     ;...
                MPj.Phi     ;...
                MPj.r2/2	;...
                MPj.r3/6	;...
                MPj.rho     ;...
                MPj.sight   ;...
                MPj.sight/sqrt(1-MPj.rho^2)   ;...
                ];
        end
        
    else                % if not re-computing standard errors, load saved ones
        
        sefnm = ['ses_' fnmsuf];
        load(sefnm)
        nse = size(psim,2);
    end
    
    pse = std(psim,[],2);   % standard deviation of parameters estimated on simulated data sets
    CI = [prctile(psim,(100-CIpct)/2,2),prctile(psim,(100+CIpct)/2,2)];     % confidence interval
    
end

%% Display results

% ParsOut: cell array holding displayed parameters

% load in point estimates
ParsOut = { 'omega',    MP.om       ;...
            'gamma',    MP.gam      ;...
            'psi',      MP.psi      ;...
            'phi_e',    MP.phi_e	;...
            'phi',      MP.phi      ;...
            'Phi',      MP.Phi      ;...
            'vrho2',    MP.r2/2     ;...
            'vrho3',    MP.r3/6     ;...
            'rho',      MP.rho      ;...
            'sigma',    MP.sight	;...
            'sd(mu)',   MP.sight/sqrt(1-MP.rho^2)	;...
          };
      
ParsOut = [ParsOut, num2cell([pse,CI])];        % include s.e.'s/CIs

% column headings
ParsOut = [{'Parameter', 'Point Est.', 's.e.', ...
    [num2str(CIpct) '% CI LB'], [num2str(CIpct) '% CI UB']}; ParsOut];

% clleig: cell array holding eigenvalue display
clleig = num2cell(eigs);    % eigenvalues
clleig = [{'Real','Imag.','Modulus'};clleig];   % column headings

% mmout: cell array holding higher moments (model and data)
mmout = [mmnm, num2cell([MMdat.datmm,MM.mm])];
mmout = [{'Moment','Data','Model'};mmout];      % column heading

% display output
disp(' ')
disp(['Objective function: ' num2str(fnvl)])
disp(' ')
disp('Eigenvalues:')
disp(clleig)
disp(' ')
disp('Parameter estimates:')
disp(ParsOut)
disp('Moments:')
disp(mmout)


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

if ~plfig       % if not plotting figures, beep and stop here
    beep
    return;
end

% set up colors
datclr = rgb('black');
modclr = rgb('dimgray');
bcclr1 = rgb('whitesmoke');
bcclr2 = rgb('gainsboro');
tkclr = rgb('dimgray');

% set up x-axis labels and limits for spectrum plots
xtickper = [2, 4, 6, 24, 32, 40, 50, 60, 80, 128];      % vector of xtick mark locations
ntick = numel(xtickper);                                % number of xtick marks
xticklab = cell(1,ntick);                               % cell to hold xtick labels
for j = 1:ntick                                         % for each tick location
    xticklab{j} = num2str(xtickper(j),'%i');            % label = string value of xtick mark
end
xtickvec = log(xtickper);                               % get logs of tick marks
tkind = find((1./xtickper>frmn)&(1./xtickper<frmx));    % indices of tick marks that will show up in figure
xax = log(min(1./freqs,realmax));                       % get log of periodicities associated with frequencies
xl = [frmn,frmx];                                       % min and max of x-axis in frequencies
xl = log(1./min(xl([2,1]),realmax));                    % min and max of x-axis in log of periodicity

%% HOURS SPECTRUM - STOCHASTIC MODEL

% determine y-axis limits
alldat = [MM.specl(spcpl);datspecl(spcpl)];
ydiff = max(alldat);             % maximum plot value
ymx = max(alldat)+ydiff*.05;     % y-axis maximum
ymn = 0;                         % y-axis minimum
yl = [ymn,ymx];

% initialize figure and advance figure counter
figure(fign)
fign = fign + 1;
clf
set(gcf, 'Position', figpos)
hold on

% plot spectra
h1=plot(xax,datspecl,'-','LineWidth',lnwd,'Color',datclr);      % data spectrum
h2=plot(xax,MM.specl,'-','LineWidth',lnwd/2,'Color',modclr);    % model spectrum

% lines at x-axis tick marks
for j = tkind
    line([xtickvec(j),xtickvec(j)],yl,'Color',tkclr,'LineWidth',0.5,'LineStyle','--')
end
handv1 = get(gca,'Children');

% area plot to shade business cycle regions
a1 = area(([log(6),log(32)]),.995*[ymx, ymx], 'facecolor',bcclr1, 'edgecolor',bcclr1);
a2 = area(([log(32),log(50)]) ,.995*[ymx, ymx], 'facecolor',bcclr2, 'edgecolor',bcclr2);
set(gca,'Layer','top')
handv = [handv1;a1;a2];
set(gca,'Children',handv)

% axis properties
xlabel('Periodicity','FontName','Times','FontSize',fsz,'Interpreter','latex')
xlim(xl)
ylim([ymn,ymx])
set(gca,'XTick',xtickvec,'XTickLabel',xticklab,'FontName','Times','FontSize',fsz,'box','on')

% legend
hleg = legend([h1,h2],{'Data','Model'},'Location','NorthWest');
set(hleg,'FontName','Times','FontSize',fsz)

% title
title('Hours Spectrum')


%% HOURS SIMULATION - STOCHASTIC MODEL

St0d = St0;          % initial state

% call function to simulate Tsim periods (burn in)
[Stsim,~] = NKRF_Sim_fn_mex(Tsim,St0d,shk(1:Tsim),coefs,expnssim,MP,ctoff,ntrmsim);

St0d = Stsim(:,end);    % use final state from first simulation as initial state

% call function to simulate Tsim periods
[Stsim,~] = NKRF_Sim_fn_mex(Tsim,St0d,shk(1:Tsim),coefs,expnssim,MP,ctoff,ntrmsim);

Stsim = [St0d,Stsim(:,1:end-1)];    % include initial state and drop last simluated state
lsim = 100*Stsim(2,:);              % extract hours, convert to percentage

% initialize figure and advance figure counter
figure(fign)
fign = fign + 1;
clf
set(gcf, 'Position', figpos)
hold on

% plot series
plot(1:numel(lsim),lsim-mean(lsim),'LineWidth',lnwd*3/4,'Color',modclr)

% axis labels
xlabel('Period \#','FontName','Times','FontSize',fsz,'Interpreter','latex')
ylabel('Hours (\%)','FontName','Times','FontSize',fsz,'Interpreter','latex')

% axis limits
xlim([0,numel(lsim)])
ylim(1.1*[min(lsim-mean(lsim)),max(lsim-mean(lsim))])

% format
set(gca,'FontName','Times','FontSize',fsz,'box','on')

% title
title('Hours Simulation')

%% HOURS SPECTRUM - DETERMINISTIC MODEL

shkDet = zeros(size(shk));          % feeding in zeros for shocks

% get moments when feeding in zeros
[~,~,MMDet] = MFNK_Est_fn(parsopt,MMdat,shkDet,MP,OP,EstPars);

% determine y-axis limits
alldat = [MMDet.specl(spcpl);datspecl(spcpl)];
ydiff = max(alldat);             % maximum plot value
ymx = max(alldat)+ydiff*.05;     % y-axis maximum
ymn = 0;                         % y-axis minimum
yl = [ymn,ymx];

% initialize figure and advance figure counter
figure(fign)
fign = fign + 1;
clf
set(gcf, 'Position', figpos)
hold on

% plot spectra
h1=plot(xax,datspecl,'-','LineWidth',lnwd,'Color',datclr);      % data spectrum
h2=plot(xax,MMDet.specl,'-','LineWidth',lnwd/2,'Color',modclr);    % model spectrum

% lines at x-axis tick marks
for j = tkind
    line([xtickvec(j),xtickvec(j)],yl,'Color',tkclr,'LineWidth',0.5,'LineStyle','--')
end
handv1 = get(gca,'Children');

% area plot to shade business cycle regions
a1 = area(([log(6),log(32)]),.995*[ymx, ymx], 'facecolor',bcclr1, 'edgecolor',bcclr1);
a2 = area(([log(32),log(50)]) ,.995*[ymx, ymx], 'facecolor',bcclr2, 'edgecolor',bcclr2);
set(gca,'Layer','top')
handv = [handv1;a1;a2];
set(gca,'Children',handv)

% axis properties
xlabel('Periodicity','FontName','Times','FontSize',fsz,'Interpreter','latex')
xlim(xl)
ylim([ymn,ymx])
set(gca,'XTick',xtickvec,'XTickLabel',xticklab,'FontName','Times','FontSize',fsz,'box','on')

% legend
hleg = legend([h1,h2],{'Data','Model'},'Location','NorthWest');
set(hleg,'FontName','Times','FontSize',fsz)

% title
title('Hours Spectrum (Deterministic)')


%% HOURS SIMULATION - DETERMINISTIC MODEL

St0d = St0;          % initial state

% call function to simulate Tsim periods (burn in)
[Stsim,~] = NKRF_Sim_fn_mex(Tsim,St0d,shkDet(1:Tsim),coefs,expnssim,MP,ctoff,ntrmsim);

St0d = Stsim(:,end);    % use final state from first simulation as initial state

% call function to simulate Tsim periods
[Stsim,~] = NKRF_Sim_fn_mex(Tsim,St0d,shkDet(1:Tsim),coefs,expnssim,MP,ctoff,ntrmsim);

Stsim = [St0d,Stsim(:,1:end-1)];    % include initial state and drop last simluated state
lsim = 100*Stsim(2,:);              % extract hours, convert to percentage

% initialize figure and advance figure counter
figure(fign)
fign = fign + 1;
clf
set(gcf, 'Position', figpos)
hold on

% plot series
plot(1:numel(lsim),lsim-mean(lsim),'LineWidth',lnwd*3/4,'Color',modclr)

% axis labels
xlabel('Period \#','FontName','Times','FontSize',fsz,'Interpreter','latex')
ylabel('Hours (\%)','FontName','Times','FontSize',fsz,'Interpreter','latex')

% axis limits
xlim([0,numel(lsim)])
ylim(1.1*[min(lsim-mean(lsim)),max(lsim-mean(lsim))])

% format
set(gca,'FontName','Times','FontSize',fsz,'box','on')

% title
title('Hours Simulation (Deterministic)')


%% RISK PREMIUM SPECTRUM - STOCHASTIC MODEL

% determine y-axis limits
alldat = [MM.specRP(spcpl);datspecRP(spcpl)];
ydiff = max(alldat);             % maximum plot value
ymx = max(alldat)+ydiff*.05;     % y-axis maximum
ymn = 0;                         % y-axis minimum
yl = [ymn,ymx];

% initialize figure and advance figure counter
figure(fign)
fign = fign + 1;
clf
set(gcf, 'Position', figpos)
hold on

% plot spectra
h1=plot(xax,datspecRP,'-','LineWidth',lnwd,'Color',datclr);      % data spectrum
h2=plot(xax,MM.specRP,'-','LineWidth',lnwd/2,'Color',modclr);    % model spectrum

% lines at x-axis tick marks
for j = tkind
    line([xtickvec(j),xtickvec(j)],yl,'Color',tkclr,'LineWidth',0.5,'LineStyle','--')
end
handv1 = get(gca,'Children');

% area plot to shade business cycle regions
a1 = area(([log(6),log(32)]),.995*[ymx, ymx], 'facecolor',bcclr1, 'edgecolor',bcclr1);
a2 = area(([log(32),log(50)]) ,.995*[ymx, ymx], 'facecolor',bcclr2, 'edgecolor',bcclr2);
set(gca,'Layer','top')
handv = [handv1;a1;a2];
set(gca,'Children',handv)

% axis properties
xlabel('Periodicity','FontName','Times','FontSize',fsz,'Interpreter','latex')
xlim(xl)
ylim([ymn,ymx])
set(gca,'XTick',xtickvec,'XTickLabel',xticklab,'FontName','Times','FontSize',fsz,'box','on')

% legend
hleg = legend([h1,h2],{'Data','Model'},'Location','NorthWest');
set(hleg,'FontName','Times','FontSize',fsz)

% title
title('Risk Premium Spectrum')

%% MOMENTS PLOT - STOCHASTIC MODEL


% initialize figure and advance figure counter
figure(fign)
fign = fign + 1;
clf
set(gcf, 'Position', figpos)
hold on

% set up parameters of bar plot
twd = 0.9;
aa = (1-twd)/2;
bb = twd/3;
pt1 = aa+bb;
pt2 = aa+2*bb;
datx = (1:nmm) + pt1;
modx = (1:nmm) + pt2;
DG = 0.2*ones(1,3);

% determine axis limits
alldat = [MMdat.datmm;MM.mm];
xdiff = max(alldat)-min(alldat);        % maximum plot value
xmx = max(max(alldat)+xdiff*.05,0);     % x-axis maximum
xmn = min(min(alldat)-xdiff*.05,0);     % x-axis minimum

% bar plots
barh(datx,MMdat.datmm(end:-1:1),0.5,'EdgeColor',DG,'FaceColor',modclr,'LineWidth',1)
barh(modx,MM.mm(end:-1:1),0.5,'EdgeColor',DG,'FaceColor',bcclr2,'LineWidth',1)

% set vertical tick locations
ytckvc = (1:nmm)+0.5;
set(gca,'YTick',ytckvc,'YTickLabel',[],'FontName','Times','FontSize',fsz,'box','on');

% set x-axis limits
xlim([xmn,xmx])

% add moment labels
ax = axis;
HorizontalOffset = 0.1;
for i = 1:nmm
    text(ax(1)-HorizontalOffset,ytckvc(nmm+1-i),mmnm{i},...
        'HorizontalAlignment','Right','Interpreter','latex','FontName','Times','FontSize',fsz);
end

% legend
hleg = legend('Data','Model','Location','NorthEast');
set(hleg,'FontName','Times','FontSize',fsz)

% title
title('Moments')

















