function rcind = simrcind(T,N,reclen0,explen0)
%
% This function simulates N bootstrapped recession histories (0-1 recession
% indicators).
%
% Inputs are:
% 
%   T           length of each history
%   N           number of histories to simulate
%   reclen0     vector of recession lengths in data
%   explen0     vector of expansion lengths in data
%
% Outputs:
%   rcind       rcind(:,j) is 0-1 recession indicator for j-th simulated history
%
% The timing assumption is that the recession captured by the length in
% reclen0(j) immediately precedes the expansion captured by the length in
% explen0(j). We draw full peak-to-peak cycles, i.e., if we draw cycle j,
% then in the simulated history we assume there is a recession of length
% reclen0(j) followed by an expansion of length explen0(j).
%
% For Beaudry, Galizia, and Portier, Putting the Cycle Back into Business
% Cycle Analysis (2019).

%% 

% we will seed random number generator for replicability
sd = 1546034615;

ncyc = size(reclen0,1);     % number of cycles in the data

rcind = zeros(T,N);         % allocate memory for output

for j = 1:N         % for each simulated history
    rng(sd+10*j,'combRecursive')        % initialize the random number generator at a new value
    
    reclenj = [];
    explenj = [];
    Ttot = 0;
    
    while Ttot < T  	% while the simulated history is less than desired length
                
        % draw 2*ncyc random integers between 1 and ncyc; each draw corresponds
        % to the index of a different cycle from the data
        cycdr = unidrnd(ncyc,2*ncyc,1);
        
        % add to recession/expansion length histories for draw j
        reclenj = [reclenj;reclen0(cycdr)];
        explenj = [explenj;explen0(cycdr)];
        
        Ttot = sum([reclenj;explenj]);  % total length of simulated history
        
    end        
    
    % call function to convert recession/expansion lengths to 0-1
    % indicators
    rcind(:,j) = rcindCreate(reclenj,explenj,T);
    
end
