% This file obtains bootstrapped confidence intervals for smoothed spectral
% densities and plots them along with point estimates. It requires the
% following files (see ReadMe.pdf for descriptions):
%
%   - simSpecCI.m
%   - rgb.m
%   - Data.mat
%
% For Beaudry, Galizia, and Portier, Putting the Cycle Back into Business
% Cycle Analysis (2019).

%% Initialize

clearvars

%% Options and parameters

Nsim = 10000;           % number of bootstrap simulations

frmn = 1/60;            % lowest frequency to plot (cycles/period)
frmx = 1/4;             % highest frequency to plot (cycles/period)

CIpct = [90,80,66];     % row vector of CI sizes (percentage)
nkr = 13;               % size of kernel for smoothing

%% Plotting options

ffign = 1;                  % first figure number to use
fsz = 16;                   % font size for axis/legend
lnwd = 5;                   % line width for point estimate
figpos = .8*[0 0 1058 817]; % figure position

%% Data options

flnm = 'Data';      % file name with data in it

% The file Data.mat contains a single struct variable called Data. Each
% field of this struct variable is another struct variable corresponding to
% a different data series (see ReadMe.pdf for list of data series). The
% fields for each variable are as follows:
%   data:       vector of actual data
%   LongName:   longer description of data series
%   fdat:       first quarter of data available for that series
%   ldat:       last quarter of data available for that series
%   m100:       flag = 1 if the data should be multiplied by 100 to convert
%                   from fractions/logs to percentages

xname = 'lhnfbBLS';     % name of data series to use; see ReadMe.pdf

% Date ranges to use; if range exceeds available range for data, code
% throws a warning and uses the maximum range available.
estfst = 0;           % first observation to use
estlst = 2015.25;     % last observation to use


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


%% Read in data and de-mean

load(flnm)      % load file containing data (also contains dates of
                % observations in the variable ddate)

x = Data.(xname).data;      % extract vector of data
m100 = Data.(xname).m100;   % extract flag = 1 if need to multiply by 100
fdat = Data.(xname).fdat;   % extract first available quarter of data
ldat = Data.(xname).ldat;   % extract last available quarter of data

dat = (fdat:.25:ldat)';     % vector of available quarters

fdat1 = estfst;             % variable to hold first quarter to use
ldat1 = estlst;             % variable to hold last quarter to use

% This block checks whether desired first quarter precedes available first
% quarter. If so, fdat1 is changed to first available quarter and a warning
% is reported.
if fdat > estfst
    yr = floor(fdat);
    qr = 4*(fdat-yr)+1;
    qrt = [num2str(yr) 'Q' num2str(qr)];
    warning(['First date precedes earliest in data. Starting at ' qrt '.'])
    fdat1 = fdat;
end
% This block checks whether desired last quarter is after available last
% quarter. If so, ldat1 is changed to last available quarter and  a warning
% is reported.
if ldat < estlst
    yr = floor(ldat);
    qr = 4*(ldat-yr)+1;
    qrt = [num2str(yr) 'Q' num2str(qr)];
    warning(['Last date after latest in data. Ending at ' qrt '.'])
    ldat1 = ldat;
end

kpind = (dat>=fdat1) & (dat<=ldat1);    % indices of quarters to keep

xc = x(kpind);          % extract subsample

% if desired, multiply by 100
if m100 == 1
    xc = 100*xc;
end

xc = xc-mean(xc);       % de-mean sample
T = numel(xc);          % length of sample


%% Spectrum parameters

TT = 1024;          % number of points to get spectrum at

% This block determines the number of points in one sided of the spectrum
Todd = logical(mod(TT,2));      % flag = true if TT is odd
if Todd                         % if TT is odd
    TT2 = (TT+1)/2;                         % number of points in one side of fft
    freqs = linspace(0,.5*(TT-1)/TT,TT2)';  % vector of ordinary frequencies, evenly spaced
else                            % otherwise if TT is even
    TT2 = TT/2 + 1;                         % number of points in one side of fft
    freqs = linspace(0,.5,TT2)';            % vector of ordinary frequencies, evenly spaced
end

kr = hamming(nkr);      % kernel function for smooothing
kr = kr/sum(kr);        % normalize weights to sum to one

plind = (freqs>=frmn) & (freqs<=frmx);  % indices of frequencies to plot
pers = 1./freqs;                        % periodicities

%% Get data spectrum

ffx = abs(fft(xc,TT)).^2;               % un-scaled two-sided spectrum
ffx = ffx(1:TT2,:)/(2*pi*T);            % scaled one-sided spectrum
conspec = [ffx(nkr:-1:2);ffx;ffx(end-1:-1:end-nkr+1)]; % extend for wrap-around kernel-smoothing
conspec = conv(conspec,kr,'same');      % kernel-smooth (convolve raw spectrum with kernel)
datspec = conspec(nkr:TT2+nkr-1,:);     % drop extra frequencies added for wrap-around

%% Obtain autocovariance function for simulations

% this block converts the one-sided smoothed spectrum to a two-sided one
if Todd
    datspecTS = [datspec;datspec(end:-1:2)];
else
    datspecTS = [datspec;datspec(end-1:-1:2)];
end

RTS = real(ifft(datspecTS*2*pi));   % invert spectrum to get ACF
R = RTS(1:T);                       % keep only first T lags

%% Bootstrap spectral densities

% call function to do bootstrapping
spcall = simSpecCI(R,Nsim,TT,TT2,kr);

%% compute bands

prctl = [(100-CIpct)/2,(100+CIpct)/2];              % percentiles for confidence intervals
bnds = datspec./prctile(spcall./datspec,prctl,2);   % confidence bands

%% Plot output

fign = ffign;

% this block determines y-axis limits
tmp = bnds(plind,:);
dat = [datspec(plind);tmp(:)];
ydiff = max(dat);               % maximum plot value
ymx = max(dat)+ydiff*.05;       % y-axis maximum
ymn = 0;                        % y-axis minimum

% initialize figure
figure(fign)                                % open figure
fign = fign + 1;
clf                                         % clear it
set(gcf, 'Position', figpos)        % set location and size
hold on                                     % set hold axis on

% settings for x-axis lables/scale
xtickper = [2, 4, 6, 24, 32, 40, 50, 60, 80, 128];   % vector of xtick mark locations
ntick = numel(xtickper);                                % number of xtick marks
xticklab = cell(1,ntick);                               % cell to hold xtick labels
for j = 1:ntick                                         % for each tick location
    xticklab{j} = num2str(xtickper(j),'%i');                % label = string value of xtick mark
end
xtickvec = log(xtickper);                               % get logs of tick marks
xax = log(min(1./freqs,realmax));           % get log of periodicities associated with frequencies
xl = [frmn,frmx];                           % min and max of x-axis in frequencies
xl = log(1./min(xl([2,1]),realmax));        % min and max of x-axis in log of periodicity

% color info
datclr = rgb('black');
CIclr = [0.55;.4;.25]*ones(1,3);
bcclr1 = rgb('whitesmoke');
bcclr2 = rgb('gainsboro');
tkclr = rgb('dimgray');

nCI = numel(CIpct);             % number of confidence intervals
ah = zeros(nCI,1);              % vector to hold area plot handles

% area plots for CIs
for m = 1:nCI
    ah(m) = fill([xax',fliplr(xax')],[bnds(:,m)',fliplr(bnds(:,m+nCI)')],CIclr(m,:));
end
set(ah,'EdgeColor','none')

h1=plot(xax,datspec,'-','LineWidth',lnwd,'Color',datclr);   % plot data spectrum

% vertical gridlines
tkind = find((1./xtickper>frmn)&(1./xtickper<frmx));
for j = tkind
    line([xtickvec(j),xtickvec(j)],[0,ymx],'Color',tkclr,'LineWidth',1,'LineStyle','--')
end
handv1 = get(gca,'Children');

% highlight business cycle regions
a1 = area(([log(6),log(32)]),.995*[ymx, ymx], 'facecolor',bcclr1, 'edgecolor',bcclr1);
a2 = area(([log(32),log(50)]) ,.995*[ymx, ymx], 'facecolor',bcclr2, 'edgecolor',bcclr2);
set(gca,'Layer','top')
handv = [handv1;a1;a2];
set(gca,'Children',handv)

% axis properties
xlabel('Periodicity','FontName','Times','FontSize',fsz,'Interpreter','latex')
xlim(xl)
ylim([ymn,ymx])
set(gca,'XTick',xtickvec,'XTickLabel',xticklab,'FontName','Times','FontSize',fsz,'box','on')

% legend
lgcl = cell(1,nCI);
for m = 1:nCI
    lgcl{m} = [num2str(CIpct(m)) '\% interval'];
end
hleg = legend(fliplr(ah),lgcl,'Location','NorthWest','FontName','Times','FontSize',fsz,'Interpreter','latex');













