function [SM,coefs,fnadd,exfl] = MFNK_Est_SimMom(shk,MP,OP)
%
% This function obtains the (non-smoothed) simulated spectra and moments
% for the desired number of simulated data sets.
% 
% Inputs:
%   shk         row vector of shock innovations
%   MP          struct variable of model parameters
%   OP          struct variable containing non-model variables/parameters
%
% Outputs:
%   SM          struct variable of spectra/moments
%   coefs       vector of coefficients on monomials in the perturbation
%                   solution
%   fnadd       variable containing penalty to add to the objective
%                   function because the parameters violate some joint
%                   restriction, or because the simulation exploded
%   exfl        flag indicating that the value in fnadd should be added to
%                   the objective function
%
% For Beaudry, Galizia, and Portier, Putting the Cycle Back into Business
% Cycle Analysis (2019).

%% Check for static uniqueness

% check to make sure the complementarities aren't so strong that they could
% create multiple equilibria; if they are, add penalty to objective 
% function and exit
if MP.Phi_ >= MP.Rp*MP.al*MP.del*(1+MP.gam-MP.psi)/(MP.tau*MP.Xi)
    exfl = -1;
    SM = [];
    coefs = [];
    fnadd = 1e6*(MP.Phi_*MP.tau*MP.Xi/(MP.Rp*MP.al*MP.del*(1+MP.gam-MP.psi))-.999);
    return;
end

% check to make sure the non-linearity isn't so strong as to create
% multiple equilibria; if it does, stop and report error (shouldn't happen
% given the parameter restrictions)
if (-MP.Dth3 < MP.Dth2^2/2)&&(abs(MP.Dth3)>1e-7)
    error('Non-unique solution')
end

%% Check for unique SS

% check to make sure the steady state is unique; if it's not, add penalty
% to objective function and exit
SSchk = 1+MP.al1*MP.psi/MP.del-MP.al2-MP.al3+3*MP.Dth2^2/(8*MP.Dth3);
if SSchk <= 0
    exfl = -1;
    SM = [];
    coefs = [];
    fnadd = 1e6*(.001-SSchk);
    return;
end

%% Get derivatives of policy function at the solution

prs = [MP.del MP.psi MP.xi MP.al2 MP.al1 MP.rho];   % reduced-form first-order parameters
Dth = [0; MP.Dth2; MP.Dth3];                        % reduced-form non-linear parameters
prsall = [prs Dth'];                                % combined reduced-form parameters

[dPhi,exfl,eigs] = NKRF_SlvMod_Full(prsall,OP);     % call function to get derivatives of policy function

if exfl == -1       % if indeterminacy detected, set fnadd to penalty amount
                        % returned from solution function and exit
    SM = [];
    coefs = [];
    fnadd = dPhi;
    return;
end

SM.eigs = eigs;     % eigenvalues of system

%% Get coefficients for simulation

% convert derivatives to Taylor coefficients
coefsslv = dPhi.*OP.cmbsslv./factorial(sum(OP.expnsslv,2));

% In the above solution, one of the state variables is 'sig', which is the
% standard deviation of the reduced-form shock innovation. This doubles as
% the perturbation parameter. Since this is constant over time, we can save
% on computational time when simulating by dropping it as a state variable
% and subsuming it into the coefficient on monomials in the remaining state
% variables. The following block computes these new coefficients. Refer to
% the original coefficients above as the "full" solution, and the new
% coefficients as the "collapsed" solution.

coefs = zeros(OP.ntrmsim,1);    % vector to hold coefficients for the collapsed solution
                                    
for j = 1:OP.ntrmsim            % for each monomial in the collapsed solution
    
    % find the indices of the monomials in the full solution that are of 
    % the same order in in the first 3 state variables as collapsed
    % monomial j
    selvc = ismember(OP.expnsslv(:,1:3),OP.expnssim(j,:),'rows');
    
    expnsslvj = OP.expnsslv(selvc,:);   % extract corresponding monomials of full solution
    coefsslvj = coefsslv(selvc);        % extract corresponding coefficients from full solution
    n = size(expnsslvj,1);              % number of such monomials
    
    % for each row of expnsslvj, multiply corresponding coeffcient by sig
    % to the relevant power and add to coefficient for the reduced solution
    % to be used for simulation
    for p = 1:n
        coefs(j) = coefs(j) + coefsslvj(p)*MP.sig^expnsslvj(p,4);
    end
end


%% Deterministic simulation to check for explosion

% call function to simulate data; if simulation explodes, it will return
% tcut > 0 (otherwise tcut=0)
[~,tcut] = NKRF_Sim_fn_mex(OP.Tsim,OP.St0,zeros(1,OP.Tsim),coefs,...
    OP.expnssim,MP,OP.ctoff,OP.ntrmsim);

if tcut         % if simulation exploded
    exfl = -1;          % set exit flag indicating problem
    fnadd = 3e5/tcut;   % penalty to add to objective function
    SM = [];
    coefs = [];
    return;             % exit
end


%% Get simulated moments

SM.speclall = zeros(OP.TT2,OP.nsim);    % matrix to hold simulated hours spectra
SM.specRPall = zeros(OP.TT2,OP.nsim);   % matrix to hold simulated risk premium spectra
SM.mmsall = zeros(OP.nmm,OP.nsim);      % matrix to hold simulated higher moments

nrem = OP.nsim;     % variable to track number of additional data sets needed;
                        % initialize to the total number needed

SM.nact = 0;        % variable to track number of data sets actually produced
fshk = 1;           % variable to track index of first shock for a simulation
fnadd = 0;          % variable to hold objective function penalty (if applicable)
while nrem > 0
    
    Tnd = nrem*OP.Tsim + OP.Tbrn;   % size of simulation to try
    
    if fshk+Tnd-1 > OP.Ttot         % if we've run out of shocks to feed in
        if OP.vrb == 1                  % if verbose output, beep and report
            beep
            display(['Full simulation unavailable. nrem = ' num2str(nrem)])
        end
        fnadd = 2e5*nrem/OP.nsim;   % compute penalty 
        break;                      % exit simulation
    end
    
    % call function to do burn-in simulation
    [Stsim,tcut] = NKRF_Sim_fn_mex(OP.Tbrn,OP.St0,shk(fshk:fshk+OP.Tbrn-1),...
        coefs,OP.expnssim,MP,OP.ctoff,OP.ntrmsim);
    
    if tcut > 0         % if burn-in exploded
        fshk = fshk+OP.Tbrn;    % re-set first shock index
        continue;               % go back to beginning of loop
    end
    
    Stin = Stsim(:,end);    % set initial state to last state from burn-in sample    
    % call function to simulate the remainder of the sample
    [Stsim,tcut] = NKRF_Sim_fn_mex(nrem*OP.Tsim,Stin,shk(fshk+OP.Tbrn:fshk+Tnd-1),...
        coefs,OP.expnssim,MP,OP.ctoff,OP.ntrmsim);

    if tcut > 0         % if simulation exploded
        
        lstt = floor(tcut)-100;     % back up 100 periods before explosion
        
        % this block calculates the number of new full data sets we were 
        % able to generate
        if lstt < OP.Tsim
            nnew = 0;
        else
            nnew = floor(lstt/OP.Tsim-1) + 1;   
        end
        
    else                % otherwise the number of new data sets is the total number needed
        nnew = nrem;
    end
    
    % if we generated at least one new data set, this block computes the
    % non-smoothed spectra and higher moments for each
    if nnew > 0
        
        lsim = reshape(Stsim(2,1:nnew*OP.Tsim)',OP.Tsim,nnew);      % simulated hours
        rpsim = 4*(MP.r1*lsim + (1/2)*MP.r2*lsim.^2 + ...           % simulated risk premium
            (1/6)*MP.r3*lsim.^3);
        
        % hours
        lsim = lsim(OP.smpl,:);             % drop unused observations
        lsim = lsim-mean(lsim,1);           % de-mean
        pfflp = abs(fft(lsim,OP.TT)).^2;    % un-scaled two-sided spectrum
        SM.speclall(:,SM.nact+1:SM.nact+nnew) = 100^2*pfflp(1:OP.TT2,:)/(2*pi*OP.Tl);     % scaled one-sided spectrum
        
        % risk premium
        rpsim = rpsim(OP.smpRP,:);          % drop unused observations
        rpsim = rpsim-mean(rpsim,1);        % de-mean
        pfflp = abs(fft(rpsim,OP.TT)).^2;   % un-scaled two-sided spectrum
        SM.specRPall(:,SM.nact+1:SM.nact+nnew) = 100^2*pfflp(1:OP.TT2,:)/(2*pi*OP.TRP);     % scaled one-sided spectrum
        

        % higher moments
        
        lsimbp = OP.BPmatl*lsim;            % BP-filtered hours
        lsimbp = lsimbp(OP.comml,:);        % drop unused observations (as in real data)
        lsimbp = lsimbp-mean(lsimbp,1);     % de-mean
        lsimscl = lsimbp./sqrt(sum(lsimbp.^2,1)/OP.TRP);    % scale by standard deviation

        rpsimbp = OP.BPmatRP*rpsim;         % BP-filtered risk premium
        rpsimbp = rpsimbp(OP.commRP,:);     % drop unused observations (as in real data)
        rpsimbp = rpsimbp-mean(rpsimbp,1);  % de-mean
        tmp = sqrt(sum(rpsimbp.^2,1)./OP.TRP);      % compute standard deviations of the risk premium
        tmp(abs(tmp)<1e-10) = 1;            % for cases where the s.d. = 0, replace with 1 (for
                                            	% version of model with
                                            	% risk premium shut down)
        rpsimscl = rpsimbp./tmp;            % scale risk premium by standard deviation
        
        % compute higher moments and reshape
        SM.mmsall(:,SM.nact+1:SM.nact+nnew) = reshape(mean( ...
            lsimscl.^OP.mmsrsh(:,1,:).*rpsimscl.^OP.mmsrsh(:,2,:),1),...
            [nnew,OP.nmm])';
        
    end
    
    SM.nact = SM.nact + nnew;       % update number of data sets actually produced
    nrem = nrem - nnew;             % number of data sets still needed
    fshk = fshk+Tnd;                % next initial shock index
end

