function MP = MFNK_Est_prtr(pars,MP,EstPars)
%
% This function takes the transformed model parameters in pars, reverses
% the transformation based on the information in EstPars, then updates the
% model parameters in MP to reflect the parameter values in pars.
% 
% Inputs:
%   pars        vector of parameter values
%   MP          struct variable of model parameters; depending on which
%                   parameters are estimated, some of these parameters will
%                   remain constant, while others will be updated based on
%                   the values in pars
%   EstPars     cell array indicating which model parameters are to be
%                   estimated, as well as any parameter transformations to
%                   be made (to capture parameter restrictions)
%
% Outputs:
%   MP          updated model parameters
%
% For Beaudry, Galizia, and Portier, Putting the Cycle Back into Business
% Cycle Analysis (2019).

%% De-transform parameters in pars

% EstPars is a cell variable with the names of the estimated parameters.
% Second element of each row gives the restriction on variable to be
% imposed during estimation. In the following table, for each type of
% restrction, x represents an unrestricted auxiliary parameter defined on
% (-infty,infty), while y represents the structural parameter that should
% be restricted. The second column of the table shows the mapping from x to 
% y that operationalizes the restriction, while the third column shows the
% inverse mapping that obtains x from y.
%       RESTRICTION             x -> y MAPPING          y -> x MAPPING
%       0 = no restriction      x -> x,                 y -> y
%       1 = between -1 and 1    x -> (e^x-1)/(1+e^x),	y -> log((1+y)/(1-y))
%       2 = less than 1         x -> 1-e^x,             y -> log(1-y)
%       3 = greater than 1      x -> 1+e^x,             y -> log(y-1)
%       4 = less than 0         x -> -exp(x),           y -> log(-y)
%       5 = between 0 and 1     x -> e^x/(1+e^x)        y -> log(y/(1-y))
%       6 = >=0                 x -> x^2,               y -> sqrt(y)

nprEst = size(EstPars,1);       % number of estimated parameters

if nprEst > 0       % if some parameters are being estimated
    EstParNm = EstPars(:,1);    % extract names of parameters
else                % otherwise just set EstParNm to an empty vector
    EstParNm = [];
end

for j = 1:nprEst    % for each estimated parameter
    prT = pars(j);          % transformed initial paramater value
    switch EstPars{j,2}     % depending on variable indicating restriction type,
                                    % do corresponding de-transformation
        case 0
            prNT = prT;
        case 1
            prNT = (exp(prT)-1)/(1+exp(prT));
        case 2
            prNT = 1 - exp(prT);
        case 3
            prNT = 1 + exp(prT);
        case 4
            prNT = -exp(prT);
        case 5
            prNT = exp(prT)/(1+exp(prT));
            prNT = max(min(prNT,1- 1e-10),1e-10);
        case 6
            prNT = prT^2;
    end
    MP.(EstPars{j,1}) = prNT;	% update parameter in MP
end

%% Update remaining parameters

if any(strcmp(EstParNm,'gam_'))     % if estimating gamma
    MP.gam = MP.gam_*(1-MP.del)*.99;    % use auxiliary estimation parameter gam_ to get gam
end

if any(strcmp(EstParNm,'psi_'))     % if estimating psi
    MP.psi = MP.psi_*(1-MP.del)*.99;    % use auxiliary estimation parameter psi_ to get psi
end

if any(strcmp(EstParNm,'Phi_'))     % if estimating Phi
    MP.Phi = MP.Phi_*(1-MP.phi)/MP.phi;    % use auxiliary estimation parameter Phi_ to get Phi
end

MP.Xi = (1-MP.phi)*MP.e/(MP.e+(1-MP.e)*MP.phi);
MP.tau = (1-MP.gam)*(MP.del+MP.psi)/MP.om;
MP.kap = MP.al*MP.del*(1+MP.gam-MP.psi)+MP.tau*MP.Xi;
MP.r1 = -(1-MP.phi+MP.phi*MP.Phi)*MP.e/(MP.e+(1-MP.e)*MP.phi)^2;

if any(strcmp(EstParNm,'r3til_'))	% if estimating vrho3 (3rd-order risk-premium coefficient)
                                        % use auxiliary parameter r3til_ to get other auxiliary
                                        % parameter r3til (used to get r3 below, which in turn 
                                        % gives vrho3)
    MP.r3til = (1-MP.phi)*MP.r2til^2/(2*(MP.kap/MP.tau+MP.r1)) + MP.r3til_;     
end
MP.r2 = (1-MP.phi)*MP.r2til;
MP.r3 = (1-MP.phi)*MP.r3til;

if any(strcmp(EstParNm,'phi_e_'))   % if estimating phi_e
    MP.phi_e = MP.phi_e_*MP.al*MP.del/MP.tau;   % use auxiliary parameter phi_e_ to get phi_e
end

den = 1+MP.tau*MP.r1/MP.kap;
MP.sig = MP.tau*MP.sight/(MP.kap*den);
MP.al1ht = MP.al*MP.del^2*(1-MP.del-MP.gam)/(MP.kap*(1-MP.del));
MP.al2ht = MP.al*MP.del*MP.gam*(1-MP.del-MP.psi)/(MP.kap*(1-MP.del));
MP.al3ht = (MP.al*MP.del-MP.tau*MP.phi_e)/MP.kap;
MP.al1 = MP.al1ht/den;
MP.al2 = MP.al2ht/den;
MP.al3 = MP.al3ht/den;
MP.xi = 1/MP.al3;
MP.Dth2 = -MP.tau*MP.r2/(MP.kap*den);
MP.Dth3 = -MP.tau*MP.r3/(MP.kap*den);