function [rpr,bnds] = fnRegr(rcind,prctl,kvc,x)
%
% This function computes point estimates and CIs for regression
% probabilities. Inputs are:
% 
%   rcind       vector of recession indicators
%   prctl       nCI-by-2 matrix, each row of which gives lower/upper bound
%                   percentiles for a CI; leave empty for no CIs
%   kvc         vector of horizons k to consider
%   x           window around t+k (must be scalar)
%
% Outputs:
%   rpr         point estimates for recession probabilities at each horizon
%   bnds        lower/upper bounds of CIs; bnds(q,:,m) are bounds for q-th
%                   horizon and m-th CI size
%
% For Beaudry, Galizia, and Portier, Putting the Cycle Back into Business
% Cycle Analysis (2019).

%% Set-up

T = numel(rcind);               % number of periods in sample
nCI = size(prctl,1);            % number of CIs to consider

zprc = norminv(prctl/100,0,1);  % convert CI percentiles to z-scores

nk = numel(kvc);                % number of horizons

% allocate memory for outputs
rpr = zeros(nk,1);
bnds = zeros(nk,2,nCI);

wn = ones(2*x+1,1);    % vector of ones equal to total window size (2x+1)

% t-th element of rcwn is 1 iff date t has a recession at some point in
% x-quarter window around it
rcwn = min(conv(rcind,wn,'same'),1);

%% Compute recession probabilities and CIs

for q = 1:nk         % for each horizon k = kvc(q)
    
    k = kvc(q);
    
    Tk = T-k-x;     % last date t of sample for which entire t+k+/-x window
                        % is in the data
	rcindk = rcind(1:Tk);   % consider only these elements of rcind
    
    % take corresponding elements of rcwn, i.e, t-th element of rcwnk
    % corresponds to (t+k)-th element of rcwn
    rcwnk = rcwn(k+1:end-x);
    
    X = [ones(Tk,1),rcindk];    % matrix of regressors
    bet = (X'*X)\(X'*rcwnk);    % regression point estimates
    e = ones(2,1);
    rprq = e'*bet;              % point estimate of recession probability
    rpr(q) = rprq;              % load into output vector
    
    if nCI > 0
        u = rcwnk-X*bet;            % vector of residuals
        
        Xu = X.*u;                  % product of regressors with residuals
        Q = NewWest(Xu,1,1);        % call function to get Newey-West estimate of
        %   covariance matrix fo sample mean of rows of Xu
        cvbet = Tk*((X'*X)\Q/(X'*X));   % HAC estimate of cov(bet)
        
        sebet = sqrt(e'*cvbet*e);   % standard error estimate for rprq
        for i = 1:nCI               % for each CI size
            bnds(q,:,i) = rprq + sebet*zprc(i,:);   % get CI bounds
        end
    end
end

