/*******************************************************************************

Murphy-Topel standard errors


This command calculates the Murphy-Topel standard errors.

*******************************************************************************/

capture program drop MT_errors
program define MT_errors, eclass

	syntax [, model(string) vmodel(string) outcome(string) modelcovars(string) delta(string) orig(string) absorb(varlist) obsage(real 24)]

	display "MT errors started"
	
	*** DROP VARIABLES (in case past runs broke) ***
		
		foreach var in fyob? fyob?? fyob_* age? age?? s2 coef* const sample {
			capture drop `var'
		}
	
	* Mark sample
	
	quietly gen byte sample=e(sample)
	
	*** ADDITIONAL VARIABLES ***
	
		quietly tabulate fyob if sample==1, generate(fyob)
		quietly tabulate age_at_move if sample==1, generate(age)
	
		quietly sum fyob if sample==1
	
		global max_fyob=r(max)
	
	*** MODEL 2 RESULTS ***
	
		* Coefficients and variance-covariance matrix
	
		matrix b = e(b)
		matrix V2 = (e(df_r)/e(N))*e(V)
	
		global M2_size=rowsof(V2)
	
		* Scores
			
		predict double res if sample==1, res
		quietly gen double res2=res^2 if sample==1
		quietly sum res2 if sample==1
		scalar mse=r(mean)
		
		quietly gen double s2=res*(1/mse) if sample==1
			
		drop sample res res2
	
	*** MODEL 1 RESULTS ***
	
		append using "`vmodel'", gen(appended)
		
		mkmat V* if V1!=., matrix(V1)
		drop V*
		quietly drop if appended==1
		drop appended
		
		global M1_size=rowsof(V1)
	
	*** SUM OF COEFFICIENTS ***
	
		quietly gen coef_orig=0
		quietly gen coef_dest=0
		
	*** COHORT INTERACTIONS *** 
		
		* Delta
		
		quietly forvalues fyob=1978(1)${max_fyob} {
			gen tmp_fyob`fyob'_delta=_b[`fyob'.fyob#c.`delta']
		}
		
		* Origin outcomes
		
		if "`model'"!="fe_I" & "`model'"!="fe_II" {			
			quietly forvalues fyob=1978(1)${max_fyob} {
				gen tmp_fyob`fyob'_orig=_b[`fyob'.fyob#c.`orig']
			}	
		}
		
		if "`model'"=="fe_I" | "`model'"=="fe_II" {
			quietly forvalues fyob=1978(1)${max_fyob} {
				gen tmp_fyob`fyob'_orig=0
			}	
		}
		
		quietly forvalues fyob=1978(1)${max_fyob} {
			local f=`fyob'-1977
			replace coef_orig = coef_orig + fyob`f' * (tmp_fyob`fyob'_orig - tmp_fyob`fyob'_delta)
			replace coef_dest = coef_dest + fyob`f' * (tmp_fyob`fyob'_delta)
		}
	
		drop tmp*
	
	*** MODEL SPECIFIC ***
	
		* Baseline I model
		
		if "`model'"=="base_I" | "`model'"=="fe_I"  {

			local start_age=2+(`obsage'-24)
		
			quietly forvalues age=`start_age'(1)34 {
				gen tmp_age`age'_delta=_b[`age'.age_at_move#c.`delta']
			}
		
			quietly forvalues age=`start_age'(1)34 {
				* Note a1 is always the bottom age
				local a=1+`age'-`start_age'
				replace coef_orig = coef_orig - age`a' * (tmp_age`age'_delta)
				replace coef_dest = coef_dest + age`a' * (tmp_age`age'_delta)
			}
	
			drop tmp*
		
		}
		
		* Baseline II model 

		if "`model'"=="base_II" | "`model'"=="fe_II" {
			#delimit ;
			quietly replace coef_orig = coef_orig 
					- early_exposure*_b[c.early_exposure#c.`delta']
					- late_exposure*_b[c.late_exposure#c.`delta']
					- post_exposure*_b[c.post_exposure#c.`delta']
					- (older==0)*_b[0.older#c.`delta']
					- (older==1)*_b[1.older#c.`delta'];
			
			quietly replace coef_dest = coef_dest 
					+ early_exposure*_b[c.early_exposure#c.`delta']
					+ late_exposure*_b[c.late_exposure#c.`delta']
					+ post_exposure*_b[c.early_exposure#c.`delta']
					+ (older==0)*_b[0.older#c.`delta']
					+ (older==1)*_b[1.older#c.`delta'];
			#delimit cr	
		}
		
		* Baseline linear model

		if "`model'"=="base_IIlinear" {		
			#delimit ;
			quietly replace coef_orig = coef_orig 
					- exposure*_b[c.exposure#c.`delta']
					- post_exposure*_b[c.post_exposure#c.`delta']
					- (older==0)*_b[0.older#c.`delta']
					- (older==1)*_b[1.older#c.`delta'];
				
			quietly replace coef_dest = coef_dest 
					+ exposure*_b[c.exposure#c.`delta']
					+ post_exposure*_b[c.post_exposure#c.`delta']
					+ (older==0)*_b[0.older#c.`delta']
					+ (older==1)*_b[1.older#c.`delta'];
			#delimit cr	
		}
		
		* Cohort study
		
		if "`model'"=="cohort_simul" {	
			
			quietly forvalues t=1(1)13 { 
				gen coef_orig`t'=0
				gen coef_dest`t'=0
			
				#delimit ;
				replace coef_orig`t' = coef_orig`t' 
					+ _b[`orig'`t']
					- early_exposure*_b[c.early_exposure#c.`delta'`t']
					- late_exposure*_b[c.late_exposure#c.`delta'`t']
					- post_exposure*_b[c.post_exposure#c.`delta'`t']
					- (older==0)*_b[0.older#c.`delta'`t']
					- (older==1)*_b[1.older#c.`delta'`t'];
			
				replace coef_dest`t' = coef_dest`t' 
					+ early_exposure*_b[c.early_exposure#c.`delta'`t']
					+ late_exposure*_b[c.late_exposure#c.`delta'`t']
					+ post_exposure*_b[c.post_exposure#c.`delta'`t']
					+ (older==0)*_b[0.older#c.`delta'`t']
					+ (older==1)*_b[1.older#c.`delta'`t'];
				#delimit cr	
			}
			
			* Cohort terms apply only to t=7

			quietly forvalues fyob=1978(1)1991 {
				gen tmp_fyob`fyob'_orig=_b[`fyob'.fyob#c.`orig'7]
				gen tmp_fyob`fyob'_delta=_b[`fyob'.fyob#c.`delta'7]
			}
		
			quietly forvalues fyob=1978(1)1991 {
				local f=`fyob'-1977
				replace coef_orig7 = coef_orig7 + fyob`f' * (tmp_fyob`fyob'_orig - tmp_fyob`fyob'_delta)
				replace coef_dest7 = coef_dest7 + fyob`f' * (tmp_fyob`fyob'_delta)
			}
					
		}
		

	*** FORM THE MATRICES ***
	
		quietly gen const=1
		
		quietly levelsof orig_mapid, local(mapids)
		
		local end_year=1991-(`obsage'-24)
		
				quietly foreach SA4 in `mapids' {
					forvalues year=1978(1)`end_year' {
						gen tmp_orig`SA4'_fyob`year'=(orig_mapid==`SA4')&(fyob==`year')
						gen tmp_dest`SA4'_fyob`year'=(dest_mapid==`SA4')&(fyob==`year')
					
						gen tmp_orig`SA4'_fyob`year'_p=((orig_mapid==`SA4')&(fyob==`year'))*p_rank
						gen tmp_dest`SA4'_fyob`year'_p=((dest_mapid==`SA4')&(fyob==`year'))*p_rank
						
						gen tmp_orig`SA4'_fyob`year'_p2=((orig_mapid==`SA4')&(fyob==`year'))*p_rank^2
						gen tmp_dest`SA4'_fyob`year'_p2=((dest_mapid==`SA4')&(fyob==`year'))*p_rank^2
					}
				}
		
		* Standard (without last constant) *
		
		if "`absorb'"=="" {
		
			if "`outcome'"==""|"`outcome'"=="24" {
				matrix accum C_orig = tmp_orig???_fyob19?? tmp_orig???_fyob19??_p const `modelcovars' [iw=s2*s2*coef_orig], nocons 
				matrix accum C_dest = tmp_dest???_fyob19?? tmp_dest???_fyob19??_p const `modelcovars' [iw=s2*s2*coef_dest], nocons 

				matrix C=C_orig+C_dest	
			}
			
			if "`outcome'"=="p10"|"`outcome'"=="p90" {
				matrix accum C_orig = tmp_orig???_fyob19?? tmp_orig???_fyob19??_p tmp_orig???_fyob19??_p2 const `modelcovars' [iw=s2*s2*coef_orig], nocons 
				matrix accum C_dest = tmp_dest???_fyob19?? tmp_dest???_fyob19??_p tmp_orig???_fyob19??_p2 const `modelcovars' [iw=s2*s2*coef_dest], nocons 

				matrix C=C_orig+C_dest	
			}
		
		}
		
		* Standard (with last constant, have to keep when absorbing) *
		
		if "`absorb'"!="" {
			if "`outcome'"==""|"`outcome'"=="24" {
				matrix accum C_orig = tmp_orig???_fyob19?? tmp_orig???_fyob19??_p const `modelcovars' const [iw=s2*s2*coef_orig], nocons absorb(`absorb')
				matrix accum C_dest = tmp_dest???_fyob19?? tmp_dest???_fyob19??_p const `modelcovars' const [iw=s2*s2*coef_dest], nocons absorb(`absorb')

				matrix C=C_orig+C_dest	
			}
			
			if "`outcome'"=="p10"|"`outcome'"=="p90" {
				matrix accum C_orig = tmp_orig???_fyob19?? tmp_orig???_fyob19??_p tmp_orig???_fyob19??_p2 const `modelcovars' const [iw=s2*s2*coef_orig], nocons absorb(`absorb')
				matrix accum C_dest = tmp_dest???_fyob19?? tmp_dest???_fyob19??_p tmp_orig???_fyob19??_p2 const `modelcovars' const [iw=s2*s2*coef_dest], nocons absorb(`absorb')

				matrix C=C_orig+C_dest	
			}
	
		}
		
		drop tmp*
		
		* Simultaneous
		
		if "`model'"=="cohort_simul" {
			
			matrix drop C
			
			quietly forvalues t=1(1)13 {
				gen fyob_`t'=fyob+7-`t'	
				
				levelsof orig_mapid, local(mapids)
				
				quietly foreach SA4 in `mapids' {
					forvalues year=`=1978+7-`t''(1)`=1991+7-`t'' {
						gen tmp_orig`SA4'_fyob`year'=(orig_mapid==`SA4')&(fyob_`t'==`year')
						gen tmp_dest`SA4'_fyob`year'=(dest_mapid==`SA4')&(fyob_`t'==`year')
					
						gen tmp_orig`SA4'_fyob`year'_p=((orig_mapid==`SA4')&(fyob_`t'==`year'))*p_rank
						gen tmp_dest`SA4'_fyob`year'_p=((dest_mapid==`SA4')&(fyob_`t'==`year'))*p_rank
					}
				}
				
				matrix accum C_orig`t' = tmp_orig???_fyob19?? tmp_orig???_fyob19??_p const `modelcovars' [iw=s2*s2*coef_orig`t'], nocons 
				matrix accum C_dest`t' = tmp_dest???_fyob19?? tmp_dest???_fyob19??_p const `modelcovars' [iw=s2*s2*coef_dest`t'], nocons 
			
				drop tmp*
			}
			forvalues t=1(1)13 {
				matrix C=C_orig`t'+C_dest`t'
			}
		}
		
		* Some output for trouble-shooting
		
		matrix dir 
		
		if rowsof(C)==${M1_size}+${M2_size} {
			display "MT dimensionality is right"
		}
		if rowsof(C)!=${M1_size}+${M2_size} {
			display "MT dimensionality is wrong"
		}
		
		* Final MT matrix
				
		matrix C = C[`=${M1_size}+1'..`=${M1_size}+${M2_size}',1..${M1_size}]
		matrix MT = V2 + (V2 * (C*V1*C' ) * V2)
	
	*** DROP VARIABLES ***
		
		foreach var in fyob? fyob?? fyob_* age? age?? s2 coef* const {
			capture drop `var'
		}
		
	*** REPOST THE RESULTS ***

	ereturn repost b=b V=MT 
	ereturn local vcetype "Mtopel"
	ereturn display
	
	display "MT errors worked"

end
