clear all; 
close all;
clc;

%% This file estimates the main specification of the model

global Switchmergers Switchdummies Start_Switch DIROFF_Switch ...
    INSIDER_switch homebI_switch

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% (o) Provide Inputs/Parameters                                    %%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Parameters:

% Home directory:
cd '~/home';     % root folder; place auxiliary Matlab files here

% Choice of Starting Values
Start_Switch = 2;   % Choose format of starting points
% 1 if start points are from AMPL; NEED .sol files for parameters
% 2 if start points are from MNL; NEED a .csv table with starting points

% Specification Choices:
Switchmergers = 1;  % Turn on/off interaction with Merger environment
Switchdummies = 3;  % = 0: no size dummies interaction; = 3: size dummies on

DIROFF_Switch = 1; % DIROFF switch, indicates whether DIR and OFF are separate
% 1 if DIR and OFF are separate variables
% 2 if DIR and OFF are ONE variable, e.g. DIROFF

INSIDER_switch = 0; % INSIDER_switch
% 0 if no insiders interactions
% 1 if only Managers
% 2 if Managers, Directors and Officers

homebI_switch = 1; % Home Bias Interactions Switch
% 0 is off, 
% 1 is on.

%% what variables to take in:

atsvar = 'ATS';
dirvar = 'DIR';
mergervar = 'takeover_prem';
ownershipvar = 'InstOwn_perc';
insidervar = 'dum15'; 

DataInputs = {atsvar,dirvar,mergervar,ownershipvar,insidervar};

%% Input Combinations to generate tables in the paper:

% Table 5, column (1)
% Switchmergers = 0;  INSIDER_switch = 0; atsvar = 'ATS'; dirvar = 'DIR';
% Table 5, column (2)
% Switchmergers = 1;  INSIDER_switch = 0; atsvar = 'ATS'; dirvar = 'DIR';
% Table 5, column (3)
% Switchmergers = 1;  INSIDER_switch = 1; atsvar = 'ATS'; dirvar = 'DIR';

% Table 6, column (1)
% Switchmergers = 0;  INSIDER_switch = 0; atsvar = 'PillStat'; dirvar = 'DIR';
% Table 6, column (2)
% Switchmergers = 1;  INSIDER_switch = 0; atsvar = 'PillStat'; dirvar = 'Dead';
% Table 6, column (3)
% Switchmergers = 1;  INSIDER_switch = 0; atsvar = 'PillStat'; dirvar = 'ExtremeS';

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%  (i) load the data (using the script PrepareData.m) and generates a .dat file %%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

global XCMATu XPMAT T PInc XXP

% call data function
[XCMATu, XPMAT, med, big] = PrepareData(DataInputs);

%% Declare Globals derived from data or initialized

global NCS NP NT NALT Pvarn Cvarn Int Int2 Int3 Int3_1 HomebInt NhomebInt

% parameters
NP = max(XCMATu(:,1)); % number of firms
NT = max(XCMATu(:,2)); % number of years
NALT = max(XPMAT(:,2)); % number of states 
        
NCS = size(XCMATu,1); % number of choice situations (number of rows) 

% numbers of Prod-specific and Cons-specific variables.
Pvarn = size(XPMAT,2) - 2; % excluding stateid timeid
Cvarn = size(XCMATu,2) - 4; % excluding firmid timeid inc head 

% input structure of interactions of the model
% E.g. Int = [1 1;
%       1 1;
%       1 1];  % indicates which of the cons-specific variables will be interacted
             % this matrix has dimensions Pvarn x Cvarn, and element (i,j) indicates whether
             % p-variable i is interacted with c-variable j    

Int = ones(Pvarn,Cvarn);
    
Int2 = sum(Int,1);
Int3 = sum(Int2);
Int3_1 = Pvarn*Cvarn;

HomebInt = [0;1;1;0];
NhomebInt = sum(HomebInt);

%%
% create vector NCS of indices that for each choice situation i,t whether t is in T1(i)
% that is, whether i cannot be subject to inertia at time t.

T1 = zeros(NCS,1);
% there is no previous year

T2 = zeros(NCS,1);
% indicates when i could be subject to inertia but switches

T3 = zeros(NCS,1);
% indicates when i could be subject to inertia and does not switch

% setting the values for the first year for the first firm in the database
T1(1) = 1; 
T2(1) = 0; 
T3(1) = 0;  

for x = 2:NCS
    T2(x) = (XCMATu(x,2) > 1)*( XCMATu(x,2) == XCMATu(x-1,2) + 1 )*( XCMATu(x,1) == XCMATu(x-1,1))*( XCMATu(x,3) ~= XCMATu(x-1,3)); 
    % year is greater than 1 & in row before year is -1 & in row before same firm id & in year before different inc
    T3(x) = (XCMATu(x,2) > 1)*( XCMATu(x,2) == XCMATu(x-1,2) + 1 )*( XCMATu(x,1) == XCMATu(x-1,1))*( XCMATu(x,3) == XCMATu(x-1,3));
    % same as former line except in year before same inc
    T1(x) = (T2(x)+T3(x) == 0);
end

nT2 = sum(T2); 

T = [T1 T2 T3];

%% Create Matrix of Presence and PastInc for all firms in database

Pres = zeros(NP,NT);    % initialize empty matrix
Incorp = zeros(NP,NT);

for j = 1:NP
    for t = 1:NT
        h = 1;
        while (XCMATu(h,1)~=j || XCMATu(h,2)~=t) && h<NCS
            h = h+1;
        end
        Pres(j,t) = (h < NCS); % fill in with 1 if found firm j at time t
        Incorp(j,t) = (h < NCS)*XCMATu(h,3);   % fill in with incorp code if found j at time t
    end
end

Pres(XCMATu(NCS,1),XCMATu(NCS,2)) = 1;
Incorp(XCMATu(NCS,1),XCMATu(NCS,2)) = XCMATu(NCS,3);

PrevIncorp = zeros(NP,NT);
PrevIncorp(:,2:20) = Incorp(:,1:19);

PInc = [];

for p = 1:NP
    PInc = [PInc; PrevIncorp(p,Pres(p,:)>0)'];
end

PInc(PInc==0) = 1; % set to 1 if no PInc; this does not matter, since if no PInc, it does not enter in likelihood.

% take only NCS in which XCMATu has relevant years
T1=T1((XCMATu(:,2)>=2),:);
T2=T2((XCMATu(:,2)>=2),:);
T3=T3((XCMATu(:,2)>=2),:);
PInc = PInc(XCMATu(:,2)>=2);
med = med(XCMATu(:,2)>=2);
big = big(XCMATu(:,2)>=2);

XCMATu = XCMATu((XCMATu(:,2)>=2),:);
NCS = size(XCMATu,1); % number of choice situations
XCMATu(:,2) = XCMATu(:,2)-1*ones(NCS,1);

NP = max(XCMATu(:,1));
NT = 19;
        
NInt = sum(sum(Int));
NCS = size(XCMATu,1); % number of choice situations

XP = reshape(XPMAT(:,3:(2+Pvarn)),NALT,NT,Pvarn);
XXP = reshape(XP,NT*NALT,Pvarn);

%% Starting points

StartInputs = {Switchmergers,Switchdummies};
StartingParams = StartParams(StartInputs);

if Start_Switch == 2

% Choose Starting Values for Mu and Sigma    
startMu = 0;
startSigma = 0;
StartingParams = [StartingParams';startMu;startSigma];

elseif Start_Switch == 1

cd 'StartParams';
    
betaCStart = csvread('betaC.sol');
betaPStart = csvread('betaP.sol');
betaHbi = csvread('betaHbi.sol');
homebStart = csvread('homeb.sol');
FEStart = csvread('FE.sol');
FEStart = FEStart(2:length(FEStart));
muStart = csvread('mu.sol');
sigmaStart = csvread('sigma.sol');

StartingParams = [betaPStart;betaCStart;betaHbi;FEStart;homebStart;muStart;sigmaStart];
    
end

startp = sum(T2)/(nT2+sum(T3)); % starting point for parameter pi

startP =StartingParams(1:Pvarn);
startCa =StartingParams(Pvarn+1:Pvarn+Int3_1);
startC = reshape(startCa',Pvarn,Cvarn);
startHbi = StartingParams(Pvarn+Int3_1+1:Pvarn+Int3_1+NhomebInt);

startH = StartingParams(Pvarn+Int3_1+NALT+NhomebInt);
startFE =[0,StartingParams(Pvarn+Int3_1+1+NhomebInt:Pvarn+Int3_1+NALT-1+NhomebInt)'];

startmu = StartingParams(Pvarn+Int3_1+NALT+1+NhomebInt);
starts = StartingParams(Pvarn+Int3_1+NALT+2+NhomebInt);

startHbi2 = zeros(Cvarn,1);
startHbi2(HomebInt==1) = startHbi;

%% CREATE .DAT FILE FOR AMPL

cd '~/home';     % root folder; place auxiliary Matlab files here

Datfilename = 'inatt_hbi.dat';

fid = fopen(Datfilename, 'w');    % this generates the data file
    fprintf (fid, '# Data generated: %s\n\n', datestr(now));  
    fprintf (fid, 'data; \n\n'); 

    % 0. parameters
    fprintAmplParam(fid, 'NP', NP, 1); % number of DM
    fprintAmplParam(fid, 'NT', NT, 1); % number of choice periods
    fprintAmplParam(fid, 'NALT', NALT, 1); % number of alternatives for each choice (fixed)
    fprintAmplParam(fid, 'NCS', NCS, 1); % number of choice situations
    fprintAmplParam(fid, 'Pvarn', Pvarn, 1); % number of p-level char
    fprintAmplParam(fid, 'Cvarn', Cvarn, 1); % number of c-level char
    fprintAmplParam(fid, 'NInt', NInt, 1); % number of interaction variables

    % 1. choices
    fprintAmplParam(fid, 'DataChoice', XCMATu(:,3), 1); % NCSx1 vector of choices    
    fprintAmplParam(fid, 'PInc', PInc, 1); % NCSx1 vector of choices    

    % 1.1 Time indices of every choice instance
    fprintAmplParam(fid, 'TT', XCMATu(:,2), 1); % NCSx1 vector of time indices of choice situations
    
    % 2. covariates
    fprintAmplParam(fid, 'XC', XCMATu(:,5:(4+Cvarn)), 1); % (NTxNP)xCvarn matrix of c-level char
    fprintAmplParam(fid, 'XP', XP, 1); % NT*NALT*Pvarn matrix of p-level char
    
    % 2.1 home alternative for each DM/period
    fprintAmplParam(fid, 'CHalt', XCMATu(:,4), 1); % (NTxNP)x1 vector of consumer/time home alternative
    
    % 3. Interactions pattern
    fprintAmplParam(fid, 'Int', Int, 1); % PvarnxCvarn matrix describing interaction pattern
    
    % 3.1 Home Bias Interactions
    fprintAmplParam(fid, 'HomebInt', HomebInt, 1); % PvarnxCvarn matrix describing interaction pattern
    
    % 4. Time/choice switching indices
    fprintAmplParam(fid, 'T1', T1, 1); % NCS vector of period indicators t1
    fprintAmplParam(fid, 'T2', T2, 1); % NCS vector of period indicators t2
    fprintAmplParam(fid, 'T3', T3, 1); % NCS vector of period indicators t3
    
    fprintAmplParam(fid, 'nT2', nT2, 1); % number of ones in T2

    % 5. Starting points
    fprintAmplParam(fid, 'startP', startP, 1); % vector of starting points for betaP
    fprintAmplParam(fid, 'startC', startC, 1); % vector of starting points for betaC
    fprintAmplParam(fid, 'startHbi', startHbi2, 1); % vector of starting points for betaC
    fprintAmplParam(fid, 'startH', startH, 1); % vector of starting points for betaC
    fprintAmplParam(fid, 'startp', startp, 1); % starting point for p
    fprintAmplParam(fid, 'startFE', startFE, 1); % vector of starting points for FE
    fprintAmplParam(fid, 'startmu', startmu, 1); % vector of starting points for mu
    fprintAmplParam(fid, 'starts', starts, 1); % vector of starting points for sigma

fclose(fid);    

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% (ii) estimate the model calling AMPL/Knitro                  %%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

!(ampl inatt_hbi_1.run) > inatt_hbi_1.out
!(knitroampl LogLikMax -AMPL wantsol=1) > knitroinatt_inatt.out
!(ampl inatt_hbi_2.run) > inatt_hbi_2.out

%% collect parameters

betaCEst = csvread('betaC.sol'); 
betaHbiEst = csvread('betaHbi.sol'); 

% the order of the coefficients of interactions is the order of the
% elements of the matrix Int when read column by column.
betaPEst = csvread('betaP.sol');
homebEst = csvread('homeb.sol');

FEEst = csvread('FE.sol');
FEEst = FEEst(2:length(FEEst));

muEst = csvread('mu.sol');
sigmaEst = csvread('sigma.sol');

ParamEst = [betaPEst;betaCEst;betaHbiEst;FEEst;homebEst;muEst;sigmaEst];

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% (iii) compute standard errors                                %%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

options = optimoptions(@fminunc,'MaxIter',1,'MaxFunEvals',1, 'FinDiffType', 'central',...
    'Algorithm','quasi-newton','GradObj','on','DerivativeCheck','off');
[ParamEst2,fval2,exitflag2,output2,grad2,hes] = fminunc(@LogLikG_mh,ParamEst,options);

%% Outer grad

OGrad = OuterGradh(ParamEst);
BHHH = zeros(length(ParamEst));
for i = 1:NCS
    BHHH =  OGrad(:,i)*OGrad(:,i)' + BHHH;
end

%% Robust Standard Errors

RobInvHess = inv(hes)*BHHH*inv(hes);

se_rob = sqrt(diag(RobInvHess));
csvwrite('se.sol',se_rob);

t_rob = ParamEst2 ./ se_rob;
csvwrite('t.sol',t_rob);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Average Preferences                                          %%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% call data function
[XCMATu, ~, dum_med, dum_big, ~] = PrepData(DataInputs);

% take care of FE params
FEEst = [0;FEEst];

%% SAVE ACTUAL MKT SHARES

% compute market shares s every period
% matrix that counts how many firms inc in state i at time t
countInc = zeros(NT,NALT);
countIncSmall = zeros(NT,NALT);
countIncMed = zeros(NT,NALT);
countIncMedLrg = zeros(NT,NALT);

countIncLarge = zeros(NT,NALT);

for t = 1:NT
    for i = 1:NALT
        countInc(t,i) = length(XCMATu(logical((XCMATu(:,3) == i).*(XCMATu(:,2) == t)),:));
        countIncSmall(t,i) = length(XCMATu(logical((dum_med==0).*(dum_big==0) .*(XCMATu(:,3) == i).*(XCMATu(:,2) == t)),:));
        countIncMed(t,i) = length(XCMATu(logical((dum_med .*(XCMATu(:,3) == i).*(XCMATu(:,2) == t))),:));        countIncMed(t,i) = length(XCMATu(logical((dum_med .*(XCMATu(:,3) == i).*(XCMATu(:,2) == t))),:));
        countIncLarge(t,i) = length(XCMATu(logical(dum_big .*(XCMATu(:,3) == i).*(XCMATu(:,2) == t)),:));
                countIncMedLrg(t,i) = countIncMed(t,i)+countIncLarge(t,i);

    end
end

totInc = countInc*ones(NALT,1); % vector of total incorporations every period
totIncSmall = countIncSmall*ones(NALT,1); % vector of total incorporations every period
totIncMed = countIncMed*ones(NALT,1); % vector of total incorporations every period
totIncLarge = countIncLarge*ones(NALT,1); % vector of total incorporations every period
totIncMedLrg = totIncMed+totIncLarge;

AmktSh = (countInc./repmat(totInc,1,NALT))';    % NALTxNT matrix of data/actual market shares
AmktShSmall = (countIncSmall./repmat(totIncSmall,1,NALT))';    % NALTxNT matrix of data/actual market shares
AmktShMed = (countIncMed./repmat(totIncMed,1,NALT))';    % NALTxNT matrix of data/actual market shares
AmktShLarge = (countIncLarge./repmat(totIncLarge,1,NALT))';    % NALTxNT matrix of data/actual market shares
AmktShMedLrg = (countIncMedLrg./repmat(totIncMedLrg,1,NALT))';    % NALTxNT matrix of data/actual market shares


%% Return average law preferences for
% avg firm,
% avg DE firm,
% avg NV firm.

if Switchdummies == 0;

    aveImpact = betaPEst + reshape(betaCEst,Pvarn,Cvarn)*mean(XCMATu(:,5:end))';
    aveImpactDE = betaPEst + reshape(betaCEst,Pvarn,Cvarn)*mean(XCMATu(XCMATu(:,3)==9,5:end))';
    aveImpactNV = betaPEst + reshape(betaCEst,Pvarn,Cvarn)*mean(XCMATu(XCMATu(:,3)==34,5:end))';
    aveImpactLarge = betaPEst + reshape(betaCEst,Pvarn,Cvarn)*mean(XCMATu((dum_big==1),5:end))';
    aveImpactSmall = betaPEst + reshape(betaCEst,Pvarn,Cvarn)*mean(XCMATu(logical((dum_med==0).*(dum_big==0)),5:end))';

['aveImpact ' 'aveImpactDE ' 'aveImpactNV ' 'aveImpactLarge ' 'aveImpactSmall']
[aveImpact aveImpactDE aveImpactNV aveImpactLarge aveImpactSmall ]

Average=[aveImpact aveImpactDE aveImpactNV aveImpactLarge aveImpactSmall ]; 

filename='Average2.sol';
fileID = fopen(filename,'w');
fprintf(fileID, '%s','aveImpact aveImpactDE aveImpactNV aveImpactLarge aveImpactSmall');
fprintf(fileID, 'ATS: %f\n DIR: %f\n OFF: %f\n', Average(1,:),Average(2,:),Average(3,:));
fclose(fileID);

elseif Switchdummies >= 1;
    
    aveImpact = betaPEst + reshape(betaCEst,Pvarn,Cvarn)*mean(XCMATu(:,5:end))';
    aveImpactDE = betaPEst + reshape(betaCEst,Pvarn,Cvarn)*mean(XCMATu(XCMATu(:,3)==9,5:end))';
    aveImpactNV = betaPEst + reshape(betaCEst,Pvarn,Cvarn)*mean(XCMATu(XCMATu(:,3)==34,5:end))';
    aveImpactLarge = betaPEst + reshape(betaCEst,Pvarn,Cvarn)*mean(XCMATu((dum_big==1),5:end))';
    aveImpactMed = betaPEst + reshape(betaCEst,Pvarn,Cvarn)*mean(XCMATu((dum_med==1),5:end))';
    aveImpactSmall = betaPEst + reshape(betaCEst,Pvarn,Cvarn)*mean(XCMATu(logical((dum_med==0).*(dum_big==0)),5:end))';

['aveImpact ' 'aveImpactDE ' 'aveImpactNV ' 'aveImpactLarge ' 'aveImpactMed ' 'aveImpactSmall']
[aveImpact aveImpactDE aveImpactNV aveImpactLarge aveImpactMed aveImpactSmall]

Average=[aveImpact aveImpactDE aveImpactNV aveImpactLarge aveImpactMed aveImpactSmall ]; 



if DIROFF_Switch == 1;
filename='Average.sol';
fileID = fopen(filename,'w');
fprintf(fileID,'%s\n', '          aveImpact   aveImpactDE   aveImpactNV   aveImpactLarge   aveImpactMed   aveImpactSmall');
fprintf(fileID, ' ATS:     %f     %f     %f     %f     %f     %f\n DIR:     %f     %f     %f     %f     %f     %f\n OFF:     %f     %f     %f     %f     %f     %f\n', ...
    Average(1,:),Average(2,:),Average(3,:));
fclose(fileID);
elseif DIROFF_Switch == 2
    filename='Average.sol';
fileID = fopen(filename,'w');
fprintf(fileID,'%s\n', '          aveImpact   aveImpactDE   aveImpactNV   aveImpactLarge   aveImpactMed   aveImpactSmall');
fprintf(fileID, ' ATS:     %f     %f     %f     %f     %f     %f\n DIR:     %f     %f     %f     %f     %f     %f\n OFF:     %f     %f     %f     %f     %f     %f\n', ...
    Average(1,:),Average(2,:));
fclose(fileID);
end

end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% (iv) compute counterfactuals                                %%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

Ncf = 4;

for kk = 0:Ncf
    
    % call data function
    [XCMATu, XPMAT, dum_med, dum_big, PvInc] = PrepData(DataInputs);

    % parameters
    NP = max(XCMATu(:,1)); % number of firms
    NT = max(XCMATu(:,2)); % number of years
    NALT = max(XPMAT(:,1)); % number of states 

    NCS = size(XCMATu,1); % number of choice situations (number of rows) 

    XP = reshape(XPMAT(:,3:(2+Pvarn)),NALT,NT,Pvarn);
    
    
    CFswitch = kk;

    if CFswitch ==0
        CFname = 'InSample';
    
    
    elseif CFswitch == 1

    XP(9,9:20,1) = 5*ones(1,12,1);
        CFname = 'DE_ATS_5';
    
    elseif CFswitch == 2

    XP(9,9:20,1) = 5*ones(1,12,1);
    XP(9,9:20,2:3) = 6*ones(1,12,2);

    CFname = 'DE_ATS_5_DIR_6_OFF1_6';
    
    elseif CFswitch == 3

    XP(34,9:20,:) = XP(9,9:20,:);
    CFname = 'NV_COPY_DE';
    
    elseif CFswitch == 4

    if strcmp(atsvar,'ATS1') == 1 && strcmp(dirvar,'DIR1') == 1;
    XP(9,9:20,1) = 5*ones(1,12,1);
    XP(9,9:20,2) = 6*ones(1,12,1);
    CFname = 'DE_ATS1_5_DIR1_6';
    end
    end
    
    %% SIMULATIONS
    
    % fix Seed
    seed = 000;
    rng(seed); 
    
    %% From HERE ON, everything is ONE YEAR AT THE TIME;

    % periods of the simulation:
    if CFswitch == 0
        startingY = 1995;
    else
        startingY = 2007;
    end
        
    endY = 2013;

    periods = endY-startingY+1;

    CFIncorp = zeros(NP,NT); % counterfactual incorporations in NPxNT matrix
    CFIncorp(:,1:(startingY-1995+1)) = Incorp(:,1:(startingY-1995+1));

    % initialize objects filled in by the simulation
    mktSh = zeros(NALT,periods);    % tracks total mkt share among firms that shift
    mktShSmall= zeros(NALT,periods);
    mktShMed= zeros(NALT,periods);
    mktShLarge= zeros(NALT,periods);
    mktShMedLrg= zeros(NALT,periods);

    mktShift = zeros(NALT,periods); % tracks market share out of first deciders and reincorporations
    reincorps = zeros(periods,1); % initialize vector that coutns reincorps
    newinc = zeros(periods,1); % vector that counts new firms

    %%

    for tt = startingY:endY

    clear PrevInc XProdr NPt XC XH XConr XInt XProdM XConM uti Xuti 

    t = tt - 1994;              % tracks how year of the simulation matches database year 
    t0 = t - (startingY - 1995); % denotes index of YEAR OF THE SIMULATION 

    XCMATut = XCMATu(XCMATu(:,2)==t,:); % pick firms for relevant year
    XPt = squeeze(XP(:,t,:)); % pick chars for relevant year; this is NALTxPvarn - squeeze gets rids of the third dimension

    % parameters
    NPt = length(XCMATut);
    NROWS = NPt*NALT;
    
    if t>1
    PrevInc = CFIncorp(Pres(:,t)>0,t-1); % pick incorporations in year t-1 for firms present in year t
    else
    PrevInc = PvInc(XCMATu(:,2)==t);
    end
    
    %% Generate utility indexes to recover analytical mkt shares

    XProdr = repmat(reshape(XPt,NALT,Pvarn),NPt,1);    % when XP created from data
    % order of indices: consumer, prod; dimensions are: (NPxNALT)xPvarn

    XC = XCMATut(:,5:5+(Cvarn-1));   % picks cons-level chars; NPtxCvarn
    XH = XCMATut(:,4);    % NPt vector of home state data

    XConr = reshape(XC,1,Cvarn*NPt);
    XConr = repmat(XConr,NALT,1);
    XConr = reshape(XConr,NPt*NALT,Cvarn); % dimensions are (NPtxNALT)xCvarn

    % create matrix made of columns reproducing p-level variables using in interaction
    XProdM=[];
    for i = 1:Cvarn
        XProdM =  [ XProdM XProdr(:,(Int(:,i)==1))]; % get columns according to columns of Int!
    end

    % create matrix made of columns reproducing c-level variables using in interaction
    XConM=[];
    for i = 1:Cvarn
        XConM = [ XConM repmat(XConr(:,i),1,sum(Int(:,i)))]; % make copies of c-level vars according to sum by col of Int
    end

    XInt = XProdM.*XConM;

    %% utility index

    uti = XProdr * betaPEst + XInt * betaCEst;  % generate utility index
    Xuti = reshape(uti,NALT,NPt);           % Xuti is (NALT, NCS)
    Xuti = Xuti + repmat(FEEst,1,NPt);        % add alternative-specific fixed effects

    % generate the home bias part of the utility index; this is a NALTxNCS matrix
    homeBM = zeros(NALT,NPt);

    homeBM = (homebEst*ones(NALT,NPt) + repmat(betaHbiEst'*XC(:,logical(HomebInt))',NALT,1)) .* (repmat(XH',NALT,1) == kron((1:NALT)',ones(1,NPt)));

    Xuti = Xuti + homeBM;   % add home bias


    % now create choice probabilities
    CCP = zeros(NALT,NPt); % this is a NALTxNPt matrix
    denom = zeros(NPt,1);

    ChoiceIndex = zeros(NPt,1);

    for j = 1:NPt
        denom(j) = ones(1,NALT)*exp(Xuti(:,j));
        CCP(:,j) = exp(Xuti(:,j))/denom(j);                      % logit choice probabilities if pi=1
        pd = makedist('Multinomial','Probabilities',CCP(:,j));   % create multinomial dist according to relevant CCPs
        ChoiceIndex(j) = random(pd);                             % fill in choice vector with draw from pd
    end

    %% inertia:

    % simulate inertia shocks
    p2 = rand(NPt,1); % normalized cost draw between 0 and 1 from uniform distribution (F (logistic draw)=uniform) if F is logistic CDF)

    expPayoutStay = zeros(NPt,1);
    expPayoutSwitch = zeros(NPt,1);
    pinatt = zeros(NPt,1);

    for h = 1:NPt
        if (PrevInc(h)>0)
            expPayoutStay(h) = Xuti(PrevInc(h),h); 
            expPayoutSwitch(h) = log(ones(1,NALT)*exp(Xuti(:,h))); % McFadden formula for the expectation of choosing the best alternative available
            pinatt(h) = ((1+exp((muEst-(expPayoutSwitch(h)-expPayoutStay(h)))/sigmaEst))^(-1)); 
            % probablity that random draw of costs is larger than payout form switching

        elseif PrevInc(h)==0
            pinatt(h) = 1;
        end 
    end
    
    choice = (p2<pinatt); 
    
    %% fill in actualChoice matrix according to inertia dummy times either
    % current preferred alternative or previous choice

    actualChoice1=zeros(NPt,1);

    for i = 1:NPt
        if PrevInc(i)>0 % for firms that were in the dataset the previous period, there MAY be inertia
           actualChoice1(i) = ChoiceIndex(i)*choice(i) + PrevInc(i)*(1-choice(i)); % choice at time t=1 is according to max uti
        elseif PrevInc(i) == 0 
            actualChoice1(i) = ChoiceIndex(i);
        end
    end

    %% retrieve market shares

    countInc = zeros(NALT,1);
    countIncShift = zeros(NALT,1);

    for i = 1:NALT
        countInc(i) = sum(actualChoice1==(i*ones(NPt,1)));
        countIncShift(i) = sum((PrevInc == 0).*(actualChoice1 == (i*ones(NPt,1))))+...
            sum((PrevInc > 0).*(PrevInc ~= actualChoice1).*(actualChoice1 == (i*ones(NPt,1))));      % and shifters
    end

    totInc = countInc' * ones(NALT,1); % vector of total incorporations every period
    totIncShift = countIncShift' * ones(NALT,1); 

    mktSh(:,t0) = countInc./repmat(totInc,NALT,1);    % NTxNALT matrix of market shares
    mktShift(:,t0) = countIncShift./repmat(totIncShift,NALT,1);

    reincorps(t0,1) = sum((PrevInc > 0).*(PrevInc ~= actualChoice1));
    newinc(t0,1) = sum(sum(PrevInc == 0));

    % fill in CF incorps
        for j = 1:NP
            h = 1;
                while (XCMATut(h,1)~=j || XCMATut(h,2)~=t) && h<NPt
                    h = h+1;
                end
            CFIncorp(j,t) = (h < NPt)*actualChoice1(h);   % fill in with incorp code if found j at time t
        end

        CFIncorp(NP,t) = XCMATut(NPt,3);

    %% fill in CF maktSh by size class
        
    countIncSmall = zeros(NALT,1);
    countIncMed = zeros(NALT,1);
    countIncLarge = zeros(NALT,1);
    countIncMedLrg = zeros(NALT,1);

    medt = dum_med(XCMATu(:,2)==t);
    bigt = dum_big(XCMATu(:,2)==t);
    
    for i = 1:NALT
        countIncSmall(i) = sum(((medt==0).*(bigt==0)).*(actualChoice1==(i*ones(NPt,1))));
        countIncMed(i) = sum((medt).*(actualChoice1==(i*ones(NPt,1))));
        countIncLarge(i) = sum((bigt).*(actualChoice1==(i*ones(NPt,1))));
        countIncMedLrg(i) = countIncMed(i)+countIncLarge(i);

    end

    totIncSmall = countIncSmall' * ones(NALT,1); % vector of total incorporations every period
    totIncMed = countIncMed' * ones(NALT,1); 
    totIncLarge = countIncLarge' * ones(NALT,1); 
    totIncMedLrg = totIncMed + totIncLarge;
    
    mktShSmall(:,t0) = countIncSmall./repmat(totIncSmall,NALT,1);    % NTxNALT matrix of market shares
    mktShMed(:,t0) = countIncMed./repmat(totIncMed,NALT,1);
    mktShLarge(:,t0) = countIncLarge./repmat(totIncLarge,NALT,1);    
    mktShMedLrg(:,t0) = countIncMedLrg./repmat(totIncMedLrg,NALT,1);   
    
    end
    
    %% Save simulated In Sample Mkt Shares
        
    if kk == 0
        SimulatedMktSh = mktSh;
        SimulatedMktShSmall = mktShSmall;    % NTxNALT matrix of market shares
        SimulatedMktShMed = mktShMed;
        SimulatedMktShLarge = mktShLarge;    % NTxNALT matrix of market shares
        SimulatedMktShMedLrg = mktShMedLrg;    % NTxNALT matrix of market shares
        
    end 
    
%% create and print GRAPHS
    
    % years for x axis
    for i = startingY-1:endY
        xyears(i-startingY+2) = {int2str(i)};
    end
    
    figure('color',[1 1 1],...
       'Visible','off');
    plotname1 = strcat(CFname,'_Plot.eps');
    subplot(1,2,1); plot([AmktSh(9,(startingY-1994)), mktSh(9,:)],'r','DisplayName','Simulated Market Share - Inertia')
    subplot(1,2,1); hold on; plot(AmktSh(9,(startingY-1994):(endY-1994)),'b','DisplayName','Actual Market Share')
    title('DE Market Share')
    legend('Simulated Mkt Share','Actual Mkt Share','Location','southwest')
    xlabel('year') % x-axis label
    set(gca,'XTick',1:periods)
    set(gca,'XTickLabel',str2mat(xyears))
    subplot(1,2,2); plot([AmktSh(34,(startingY-1994)),mktSh(34,:)],'r','DisplayName','Simulated Market Share - Inertia')
    subplot(1,2,2); hold on; plot(AmktSh(34,(startingY-1994):(endY-1994)),'b','DisplayName','Actual Market Share')
    title('NV Market Share')
    xlabel('year') % x-axis label
    set(gca,'XTick',1:periods)
    set(gca,'XTickLabel',str2mat(xyears))
    print( gcf, '-depsc', plotname1 )
   
    plotname4 = strcat(CFname,'_MedLrg_Plot.eps');
    subplot(1,2,1); plot([AmktShMedLrg(9,(startingY-1994)),mktShMedLrg(9,:)],'r','DisplayName','Simulated Market Share - Inertia')
    subplot(1,2,1); hold on; plot(AmktShMedLrg(9,(startingY-1994):(endY-1994)),'b','DisplayName','Actual Market Share')
    title('DE Market Share')
    legend('Simulated Mkt Share','Actual Mkt Share','Location','southwest')
    xlabel('year') % x-axis label
    set(gca,'XTick',1:periods)
    set(gca,'XTickLabel',str2mat(xyears))
    subplot(1,2,2); plot([AmktShMedLrg(34,(startingY-1994)),mktShMedLrg(34,:)],'r','DisplayName','Simulated Market Share - Inertia')
    subplot(1,2,2); hold on; plot(AmktShMedLrg(34,(startingY-1994):(endY-1994)),'b','DisplayName','Actual Market Share')
    title('NV Market Share')
    xlabel('year') % x-axis label
    set(gca,'XTick',1:periods)
    set(gca,'XTickLabel',str2mat(xyears))
    print( gcf, '-depsc', plotname4 )
        figure('color',[1 1 1],...
       'Visible','off');
   
    
end
