%PURPOSE: estimate the demand side of the smartphone model
clear; clear global; rng(1135, 'twister');
if ispc; dbstop if error;  end;

%% set up
run_for_test = false;   % if the code is only meant for a test

n_smln = 300;           % number of simulation draws
n_starts = 12;          % number of starting values
n_draws = 50;           % number of demand/mc draws when evaluating expected period profits
n_draws_fc = 5;         % num of draws for fixed cost in counterfactual simulations
Mkt_CF = 'n_mkt';       % which markets to conduct counterfactual: n_mkt, or any vectors of integers from 1 to n_mkt

tol_inner = 1e-12;      
spacing = 1e-3;         

setup.invertcost = false;

setup.pricing_model = 'sequential'; 

exr_IV = [4];       % row vector of dates of exchange rates used for IV
                    % 0: current IV, 1: last period, ...
                    % max: 12

% define model
mktsize = 30000;

Chara_w_LinearPara_in_Demand = 'screensize generation_dummies camres batterytalktime weight brandtime_dummies isflagship carriertime_dummies time_dummies price'; % chara in the utility fctn with linear parameters.
Chara_inIncludedIV1 = 'screensize generation_dummies camres batterytalktime weight';
Chara_inIncludedIV2 = 'brandtime_dummies isflagship carriertime_dummies time_dummies'; % const 1 will be added later
Chara_quality = 'screensize generation_dummies camres batterytalktime weight';
dim_quality_in_x_det = (1:8);

Chara_w_Randcoef = '';
Chara_inExcludedIV = 'ones(n_obs, 1) screensize generation_dummies camres batterytalktime weight';

setup.Var_inMC = 'carriertime_dummies brand_dummies';

ExcludedIV_demand = '[iv_same_oem_old iv_diff_oem_old]';
ExcludedIV_mc = '[]';

setup.q_scale_in_cost = 10;

% other
pfid = 1;

%% load data and define sample
varlist = {'yyyy', 'quarter', 'mm', 'quantity', 'price', ...
    'carrier','oem','model', ...
    'weight', 'dpix', 'display_width', 'display_height', 'generation',...
    'camres', 'batterytalktime',...
    'fsdate','lyear','lmonth'};
load('data\data_smartphone_clean.mat', varlist{:});

%% load flagship products list
flagship_products = load('data\flagship_products.mat');
tmp1 = strcat(flagship_products.brand, flagship_products.carrier, flagship_products.model);
tmp2 = strcat(oem, carrier, model);
tmp11 = tmp1(flagship_products.isflagship==1, :);
isflagship = ismember(tmp2, tmp11) | ismember(oem, {'Apple'});
clear tmp1 tmp2 tmp11;
varlist = [varlist, {'isflagship'}];

%% merge certain smartphones (they are considered one)
[tmp, ~, tmpind] = unique(model);
% see flagship_products_for_discussion2.xlsx for what flagship products are considered one product
ind = ismember(tmp, {'Captivate'}); tmp{ind} = 'GalaxyS';
ind = ismember(tmp, {'Epic 4G'}); tmp{ind} = 'GalaxyS';
ind = ismember(tmp, {'Fascinate'}); tmp{ind} = 'GalaxyS';
ind = ismember(tmp, {'Vibrant'}); tmp{ind} = 'GalaxyS';
ind = ismember(tmp, {'Epic 4G Touch'}); tmp{ind} = 'GalaxySII';
ind = ismember(tmp, {'Hero CDMA'}); tmp{ind} = 'HeroDeroid';
ind = ismember(tmp, {'DROID ERIS'}); tmp{ind} = 'HeroDeroid';
ind = ismember(tmp, {'EVO 4G'}); tmp{ind} = 'EVO-Inspire-ThunderBolt';
ind = ismember(tmp, {'Inspire 4G'}); tmp{ind} = 'EVO-Inspire-ThunderBolt';
ind = ismember(tmp, {'ThunderBolt'}); tmp{ind} = 'EVO-Inspire-ThunderBolt';
ind = ismember(tmp, {'Sensation 4G'}); tmp{ind} = 'Sensation-Vivid-Rezound';
ind = ismember(tmp, {'Vivid'}); tmp{ind} = 'Sensation-Vivid-Rezound';
ind = ismember(tmp, {'Rezound'}); tmp{ind} = 'Sensation-Vivid-Rezound';
ind = ismember(tmp, {'One X AT&T'}); tmp{ind} = 'OnexEvo4G';
ind = ismember(tmp, {'EVO 4G LTE'}); tmp{ind} = 'OnexEvo4G';
ind = ismember(tmp, {'Optimus T'}); tmp{ind} = 'OptimusTS';
ind = ismember(tmp, {'Optimus S'}); tmp{ind} = 'OptimusTS';
ind = ismember(tmp, {'Touch Pro2 US'}); tmp{ind} = 'Touch Pro2';

% delete carrier names from model names
tmp = strrep(tmp, 'AT&T', '');
tmp = strrep(tmp, 'T-Mobile', '');
tmp = strrep(tmp, 'Sprint', '');
tmp = strrep(tmp, 'Verizon', '');
tmp = strrep(tmp, 'CDMA', '');
tmp = strrep(tmp, ' ', '');
tmp = strrep(tmp, '-', '');

model_clean = tmp(tmpind);
varlist = [varlist, {'model_clean'}];

%% CPI
% Import the CPI data to discount all prices
[~, ~, raw] = xlsread('data\CPI_data_1_51.xlsx');
raw = raw(1:72,:);

% Create output variable
data = reshape([raw{:}],size(raw));

% Allocate imported array to column variable names
cpi = data(:,2);
cpi = cpi/cpi(1);

% Clear temporary variables
clearvars data raw;

for ny = max(yyyy)
    for nm = max(mm)
        price(yyyy==ny & mm==nm) = price(yyyy==ny & mm==nm)/cpi(12*(ny-2009)+nm);
    end
end

%% Exchange Rate
% Import exchange rates as price shifters
[~, ~, raw] = xlsread('data\exchange_rate_08_13.xlsx','A2:E73');

% Create output variable
data = reshape([raw{:}],size(raw));

% Allocate imported array to column variable names
yyyy_exr = data(:,1);
mm_exr = data(:,2);
exr = data(:, 3:5);

% normalize the variables
exr = bsxfun(@rdivide, exr, exr(1, :));

% Clear temporary variables
clearvars data raw;

% create IV for the period of 2009 to 2013 (the span of the exchange rate data)
varlist = [varlist, {'exchange_rate_IV'}];
n_exr_iv = length(exr_IV(:));
exchange_rate_IV = [];

for ne = 1 : n_exr_iv    
    exchange_rate_IV0 = zeros(size(price, 1), 3);        
    for t = 2009*12+1 : 2013*12+12
        y_t = floor((t-1)/12);
        m_t = t - y_t*12;
        t_ne = t - exr_IV(ne);
        y_t_ne = floor((t_ne-1)/12);
        m_t_ne = t_ne - y_t_ne*12;   
        ind_t = yyyy==y_t&mm==m_t;
        exchange_rate_IV0(ind_t, :) = repmat(exr(yyyy_exr==y_t_ne&mm_exr==m_t_ne, :), [sum(ind_t), 1]);
    end
    exchange_rate_IV = [exchange_rate_IV, exchange_rate_IV0];
end
clear exchange_rate_IV0

[~, ~, carrier_id] = unique(carrier);
[oem_list, ~, oem_id] = unique(oem);
[~, ~, model_id] = unique(strcat(oem, model_clean));
[~, ~, product_id] = unique(strcat(oem, carrier, model));
varlist = [varlist, {'carrier_id', 'oem_id', 'model_id', 'product_id'}];

ind_drop = false(length(yyyy), 1);
varlist = [varlist, 'ind_drop'];
ind_drop = ind_drop | (yyyy >= 2014 | (yyyy == 2013 & quarter > 1)); % drop data from March 2013 due to T-Mobile's the uncarrier compaign

% drop observations towards the end of the life of a product due to that fact that these products play a small role in the market and the data quality is low
varlist = unique(varlist); % in case a variable is listed twice
[~, sortind] = sortrows([carrier_id, product_id, yyyy, mm]);
for i = 1:length(varlist)
    eval([varlist{i} ' = ' varlist{i} '(sortind, :);']);    
end

lastrow_product = diff(carrier_id) > 0 | diff(product_id) > 0;
lastrow_product = [0; find(lastrow_product); length(carrier_id)];

sales_since_launch = nan(length(carrier_id), 1);
sales_till_termination = nan(length(carrier_id), 1);
sales_till_termination_prop = nan(length(carrier_id), 1);
for i = 1:length(lastrow_product)-1
    tmpind = lastrow_product(i) +1 : lastrow_product(i+1);

    sales_since_launch(tmpind) = cumsum(quantity(tmpind));
    prod_sales = sales_since_launch(tmpind(end));
    sales_till_termination(tmpind) = prod_sales - sales_since_launch(tmpind) + quantity(tmpind(end));
    sales_till_termination_prop(tmpind) = sales_till_termination(tmpind)/prod_sales;
end
ind_drop = ind_drop | (sales_since_launch<5 & fsdate>2009*13+1) | (sales_till_termination_prop<0.05) | (sales_till_termination<5);
clear sales_since_launch sales_till_termination

ind_drop = ind_drop | (ismember(carrier, 'AT&T')&ismember(model, 'Epic 4G Touch'))...
    | (ismember(carrier, 'T-Mobile')&ismember(model, 'Epic 4G Touch'));   % Very low sales and high prices

varlist = setdiff(varlist, 'ind_drop');
varlist = unique(varlist); % in case a variable is listed twice
for i = 1:length(varlist)
    eval([varlist{i} '(ind_drop,:) = [];']);    
end
clearvars tmpind tmpid sortind lastrow_product ind_drop ind_before_first ind_after_last ind2 ind1 i 

%% prepare the data for estimation
apple = ismember(oem, {'Apple'});
varlist = [varlist, {'apple'}];

weight = weight/100;
batterytalktime = batterytalktime/10;
camres = camres/10;
screensize = sqrt(display_height.^2 + display_width.^2) ./ dpix; % diagonal size
clearvars display_height display_width
varlist = [varlist, {'screensize'}];
varlist(ismember(varlist, {'display_height','display_width'})) = [];

% define brand_id brand dummies (we use others as the benchmark)
brand_list = {'Apple', 'Samsung', 'BlackBerry'};
brand_dummies = nan(length(yyyy), length(brand_list));
brand_id = zeros(length(yyyy), 1);
for i = 1:length(brand_list)
    ind = ismember(oem, brand_list(i));
    brand_dummies(:,i) = ind;
    brand_id(ind) = i;
end
varlist = [varlist, {'brand_dummies'}];

% define generation dummies
generation(generation==0|generation==4|generation==5) = 5; 
tmp = unique(generation); 
generation_dummies = nan(length(yyyy), length(tmp)-1); 
for i = 2:length(tmp)
    generation_dummies(:,i-1) = generation == tmp(i);
end
varlist = [varlist, {'generation_dummies'}];


year_quarter = (yyyy - min(yyyy))*4 + quarter;
year_month = (yyyy - min(yyyy))*12 + mm;    
varlist = [varlist, {'year_quarter','year_month'}];

tmp = length(year_month);
for i = 1:length(varlist)
    eval(['tf = tmp == size(' varlist{i} ',1);']);
    if ~tf; error('the sizes of variables in varlist are not consistent'); end;
end
n_obs = length(price);
setup.n_obs = n_obs;

% define "time"
time = year_month;
varlist = [varlist, {'time'}];

% define age of a product
age_in_month = nan(n_obs, 1);
age_in_quarter = nan(n_obs, 1);
uniqueproduct = unique(model_id);
for i = 1:length(uniqueproduct)
    ind = model_id == uniqueproduct(i);
    tmptmp = year_month(ind);
    age_in_month(ind) = tmptmp - min(tmptmp);
    tmptmp = year_quarter(ind);
    age_in_quarter(ind) = tmptmp - min(tmptmp);
end
age = age_in_quarter;
varlist = [varlist, {'age','age_in_quarter'}];
clear tmp tmptmp;

% define time dummies (we use the first quarter as the benchmark)
tmp = unique(quarter);
time_dummies = zeros(length(quarter), length(tmp)-1);
for i = 2:length(tmp)
    time_dummies(:,i-1) = quarter == tmp(i);
end
varlist = [varlist, {'time_dummies'}];

% define carrier dummies
[tmp, tmpind] = unique(carrier_id);
fprintf(pfid, 'carrier dummies:'); 
carrier_dummies = zeros(n_obs, length(tmp));
for i = 1:length(tmp)
    fprintf(pfid, ' %s',carrier{tmpind(i)});
    carrier_dummies(:, i) = carrier_id == tmp(i);
end
fprintf(pfid, '\n');
varlist = [varlist, {'carrier_dummies'}];

% define carrier/time dummies (we use AT&T/first-quarter as the benchmark)
carrier_year = carrier_id * 1e4 + yyyy; if any(yyyy>1e4); error('error in defining carrier_year'); end;
[tmp, tmpind] = unique(carrier_year);
carriertime_dummies = zeros(n_obs, length(tmp)-1);
for i = 2:length(tmp)
    carriertime_dummies(:,i-1) = carrier_year == tmp(i);
end
fprintf(pfid, '\n');
varlist = [varlist, {'carriertime_dummies'}];

% define brand/time dummies (we use others/first-quarter as the benchmark)
[tmp, tmpind] = unique(brand_id);
brandtime_dummies = zeros(n_obs, length(tmp)-1);
for i = 2:length(tmp)
    brandtime_dummies(:,i-1) = brand_id == tmp(i);
end
varlist = [varlist, {'brandtime_dummies'}];

% sort data
varlist = unique(varlist); % in case a variable is listed twice
[~, sortind] = sortrows([time, product_id]);
for i = 1:length(varlist)
    tf = eval(['~isempty(' varlist{i} ')']);
    if tf
        eval([varlist{i} '=' varlist{i} '(sortind, :);']);
    end
end

% define covariates in the utility function
normalization_price = 100;
price = price./normalization_price;
setup.price = price;
setup.apple = apple;

% define market structure, market shares and relative characteristics
ind = find(diff(year_month) > 0);
ind_mkt_start = [1; ind+1];
ind_mkt_end = [ind; n_obs];

n_mkt = length(ind_mkt_start);

mktid = nan(n_obs,1);
for i = 1:n_mkt
    mktid(ind_mkt_start(i):ind_mkt_end(i)) = i;
end
setup.mktid = mktid;
clear mktid


setup.isflagship = isflagship;

share = quantity ./ mktsize;


share0 = nan(n_obs, 1);
for t = 1:n_mkt
    ind_t = ind_mkt_start(t):ind_mkt_end(t);
    share0(ind_t) = 1 - sum(share(ind_t));
end

% define x_det, x_rand, IV
x_det = eval(['[' Chara_w_LinearPara_in_Demand ']']);
x_det = [x_det, ones(n_obs, 1)]; %!!! the constant has to be at the end b/c in gmmobj, we assume that x_det(:,dim_quality_in_x_det) are quality characteristics. See above, dim_quality_in_x_det assumes the frist several columns of x_det are quality characteristics
n_x_det_in_demand = size(x_det,2);

if rank(x_det) < size(x_det, 2); error('there might be some multicollinearity in x_det'); end;

x_excludediv = eval(['[' Chara_inExcludedIV ']']);
x_includediv1 = eval(['[' Chara_inIncludedIV1 ']']);
x_includediv2 = eval(['[' Chara_inIncludedIV2 ']']);
tmpind = age_in_quarter == 0 & ~(lyear<=2007 | (lyear==2008 & lmonth <= 8)); % age in quarter = 0 except those who launched before 2008Q3
x_includediv1(tmpind,:) = 0;
x_includediv = [x_includediv1, x_includediv2, ~tmpind];

x_rand = eval(['[' Chara_w_Randcoef ']']);

clear x_includediv1 x_includediv2

% construct instrument variables
iv_same_oem_old = nan(n_obs, size(x_excludediv, 2)); % sum of charactersitics (sum over other pre-existing products that (1) are of the same oem; (2) currently in the market)
iv_diff_oem_old = nan(n_obs, size(x_excludediv, 2)); % sum of charactersitics (sum over other pre-existing products that (1) are of the diff oem; (2) currently in the market)

ind_preexisting = age > 0;
for i = 1:n_obs % loop over carrier/oem/model/time
    ind_current = time == time(i);
    ind_same_oem = oem_id == oem_id(i);

    ind1 = ind_same_oem & ind_current;
    ind2 = ~ind_same_oem & ind_current;
    ind11 = ind1 & ind_preexisting;
    ind22 = ind2 & ind_preexisting;

    ind11(i) = false; % exclude this product

    if ~any(ind11)
        iv_same_oem_old(i,:) = 0;
    else
        iv_same_oem_old(i,:) = sum(x_excludediv(ind11,:), 1);
    end

    if ~any(ind22)
        iv_diff_oem_old(i,:) = 0;
    else
        iv_diff_oem_old(i,:) = sum(x_excludediv(ind22,:), 1);
    end

end
clearvars ind1 ind2 ind11 ind22 ind_same_oem ind_preexisting ind_current

clear quarter sales_till_termination_prop sortind time_dummies trialmeanutility year_month year_quarter

%% OLS and 2SLS estimation
y = log(share./share0);

% the linear projection
beta_ols = inv(x_det'*x_det)*x_det'*y;    
se_ols = sqrt(diag( mean((y-x_det*beta_ols).^2)*(inv(x_det'*x_det))));

fprintf(pfid, 'OLS\n');
fprintf(pfid, 'beta\ts.e.\n');
fprintf(pfid, '%1.3f\t%1.3f\n', [beta_ols(:), se_ols(:)]');

IV_demand = eval(ExcludedIV_demand);
IV_demand = [IV_demand x_includediv]; 
IV_demand = [IV_demand exchange_rate_IV];
if rank(IV_demand) < size(IV_demand, 2); error('there might be some multicollinearity in IV_demand'); end;
[beta_2sls, se_2sls] = gmm_iv(y, x_det, IV_demand); 

fprintf(pfid, '2SLS\n');
fprintf(pfid, 'beta\ts.e.\n');
fprintf(pfid, '%1.3f\t%1.3f\n', [beta_2sls(:), se_2sls(:)]');


%% random coefficient estimation
% IV_demand = eval(ExcludedIV_demand);
% IV_demand = [IV_demand x_includediv]; % note that the constant 1 is x_includediv already
% IV_demand = [IV_demand exchange_rate_IV];
% 
% if rank(IV_demand) < size(IV_demand, 2); error('there might be some multicollinearity in IV_demand'); end;

% IV_mc = eval(ExcludedIV_mc);
% IV_mc = [IV_mc x_includediv_mc];
% clear x_includediv_mc

clear iv_diff* iv_same* x_includediv x_excludediv

prods = ind_mkt_end - ind_mkt_start + 1; % n_mkt x 1

W = inv(IV_demand'*IV_demand);
IV = IV_demand;

n_chara_w_randcoef = size(x_rand,2) + 1; % include quality
if n_chara_w_randcoef>0
    randomdraw = norminv(net(scramble(haltonset(n_chara_w_randcoef,'Skip',1e3), 'RR2'), n_smln))';
else
    randomdraw = [];
end
setup.meanval_logit = y; % starting value for mean values 

setup.n_smln = n_smln;
setup.n_obs = n_obs;
setup.n_mkt = n_mkt;
setup.tol_inner = tol_inner;
setup.pfid = pfid;
setup.Chara_w_Randcoef = Chara_w_Randcoef;
setup.dim_quality_in_x_det = dim_quality_in_x_det;

opts = optimset('Display','iter', 'MaxIter',2*1e4,'MaxFunEvals',1e5, 'TolX', 1e-12, 'TolFun', 1e-6);  

%% define additional market structure variables
unique_oem = unique(oem_id);
n_oem = length(unique_oem);
oem_ind = false(n_obs, n_oem);
for i = 1:n_oem
   ind = oem_id == unique_oem(i);
   oem_ind(ind,i) = true;
end
oem_id_apple = unique(oem_id(ismember(oem, 'Apple')));
setup.col_in_oem_ind_for_apple = find(unique_oem == oem_id_apple);
unique_carrier = unique(carrier_id);
n_carrier = length(unique_carrier);
carrier_ind = false(n_obs, n_carrier);
for i = 1:n_carrier
   ind = carrier_id == unique_carrier(i);
   carrier_ind(ind,i) = true;
end

% define the mapping from oem_ind to brand dummies
oem_to_brand = zeros(n_oem, 1);
n_brand = length(brand_list);
for nn = 1 : n_brand    
    oem_to_brand_ind = ismember(oem_list, brand_list{nn});
    if sum(oem_to_brand_ind)>0
        oem_to_brand(nn) = find(oem_to_brand_ind);
    end
end
setup.oem_to_brand = oem_to_brand;

% define unique_ind_ijk and recover_ind_ijk for computing foc
tmp = ind_mkt_end-ind_mkt_start+1; % # of products in a mkt
[uniquenumproduct, ~, uniquenumproduct2mkt] = unique(tmp); % in computing derivatives, use cell uniquenumproduct2mkt(i_mkt) of unique_ind_ijk(r) and recover_ind_ijk(r) 
n_tmp = length(uniquenumproduct);

unique_ind_ijk = cell(n_tmp, 2);
recover_ind_ijk = cell(n_tmp, 2);
for t = 1 : n_tmp
    n_prod_t = uniquenumproduct(t);

    ind = (1:n_prod_t)';
    ind_ijk = [repmat(ind, [n_prod_t, 1]), kron(ind, ones(n_prod_t,1))];
    ind_ijk = [repmat(ind_ijk, [n_prod_t, 1]), kron(ind, ones(n_prod_t.^2,1))];

    ind_ijk = sort(ind_ijk, 2);
    [unique_ind_ijk_tmp, ~, recover_ind_ijk{t,2}] = unique(ind_ijk, 'rows');
    unique_ind_ijk_12 = sort(unique_ind_ijk_tmp(:,1:2), 2);
    [unique_ind_ijk{t,1}, ~, recover_ind_ijk{t,1}] = unique(unique_ind_ijk_12, 'rows');
    unique_ind_ijk{t,2} = unique_ind_ijk_tmp(:,3);

    if ~isequal([unique_ind_ijk{t,1}(recover_ind_ijk{t,1},:), unique_ind_ijk{t,2}], unique_ind_ijk_tmp)
        error('error in unique_ind_ijk and recover_ind_ijk');
    end
end
setup.unique_ind_ijk = unique_ind_ijk;
setup.recover_ind_ijk = recover_ind_ijk;
setup.uniquenumproduct2mkt = uniquenumproduct2mkt;

clear unique_ind_ijk  recover_ind_ijk ind_ijk unique_ind_ijk_tmp unique_ind_ijk_12 uniquenumproduct2mkt;

%% first stage GMM using inefficient weighting matrix
setup.no_transformation_of_para = false;

% starting values
para_qualitycoef_start = beta_2sls(dim_quality_in_x_det(2:end))/beta_2sls(1);
if n_starts > 1
    para_qualitycoef_start = repmat(para_qualitycoef_start, [1,n_starts]);
    para_qualitycoef_start(:,2:end) = para_qualitycoef_start(:,2:end) + randn(size(para_qualitycoef_start,1), n_starts-1);
end

para_logsigma_start = log(abs(randn(size(x_rand, 2)+1, n_starts))/10); 
para_nonlinear_start = [para_qualitycoef_start; para_logsigma_start];

% initialize
fval_reps = nan(n_starts, 1);
exitflag_reps = nan(n_starts, 1);
trialfval = nan(n_starts, 1);
trialflag = nan(n_starts, 1);
trialest = nan(size(para_nonlinear_start, 1), n_starts);
trialmeanutility = nan(n_obs, n_starts); 

nIV = size(IV,2);      

for reps=1 : n_starts
    fprintf(pfid, '====== start stage-1 estimation: starting point %1.0f ====\n', reps); t0 = cputime;
    meanval = y; % starting value for mean values
    expmeanval = exp(y);
    try
        [para_nonlinear_est, trialfval(reps), trialmeanutility(:,reps), trialflag(reps)]...
            = fminsearch_intermediate_output(@gmmobj_BLP_smartphone,...
            para_nonlinear_start(:,reps),opts,share, x_rand, x_det,...
            ind_mkt_start,ind_mkt_end, ...
            randomdraw, IV, W, setup, n_oem, n_carrier,...
            n_mkt, carrier_ind, oem_ind, [], []);
        trialest(:,reps) = para_nonlinear_est(:);

    catch err
        fprintf(pfid, 'error message = %s\n', err.message);
        for i = 1:length(err.stack)
            fprintf(pfid, '%s, Line %1.0f\n', err.stack(i).file, err.stack(i).line);
        end
        continue;
    end
    fprintf(pfid, '^^^^ done with stage-1 estimation: starting point %1.0f, time used = %1.0f ^^^^\n', reps, cputime - t0);
end

save('best_est_stage1_part1.mat');

setup.no_transformation_of_para = false;

ind1 = find(trialflag==1);
if run_for_test
    ind1 = 1;
else
    if isempty(ind1); disp('stage-1 estimation did not converge. abort the program'); return; end;
end
[~, ind2] = min(trialfval(ind1));
ind = ind1(ind2);
optimal_est_first_stage = trialest(:,ind);
meanutility00 = trialmeanutility(:,ind);
meanval = meanutility00;
expmeanval = exp(meanutility00);

[f, ~, ~, para_linear_est] = gmmobj_BLP_smartphone(optimal_est_first_stage, share, x_rand, x_det, ind_mkt_start,ind_mkt_end, randomdraw, IV, W, setup, n_oem, n_carrier, n_mkt, carrier_ind, oem_ind, meanutility00, []);
if abs(f-trialfval(ind))>1e-5; error('f and trialfval(ind) not equal'); end;

save('best_est_stage1.mat', 'optimal_est_first_stage', 'para_linear_est', 'meanutility00');

% optimal weighting matrix
[f, ~, g_obs] = gmmobj_BLP_smartphone(optimal_est_first_stage, share, x_rand, x_det,ind_mkt_start,ind_mkt_end, randomdraw, IV, W, setup, n_oem, n_carrier, n_mkt, carrier_ind, oem_ind, meanutility00, []);
if abs(f-trialfval(ind))>1e-5; error('f and trialfval(ind) not equal'); end;
Om = g_obs'*g_obs/n_obs;
W = inv(Om); 
clear g_obs

% optimal second stage
para_nonlinear_start = repmat(optimal_est_first_stage(:), [1, n_starts]);
para_nonlinear_start(:, 2:end) = para_nonlinear_start(:, 2:end) + randn(length(optimal_est_first_stage), n_starts-1)/10;

trialfval = nan(n_starts, 1);
trialflag = nan(n_starts, 1);
trialest = nan(size(para_nonlinear_start,1), n_starts);
trialmeanutility = nan(n_obs, n_starts);

for reps=1 : n_starts
    fprintf(pfid, '====== start stage-2 estimation: starting point %1.0f ====\n', reps); t0 = cputime;
    try
        [para_nonlinear_est, trialfval(reps), trialmeanutility(:,reps), trialflag(reps)]...
            = fminsearch_intermediate_output(@gmmobj_BLP_smartphone,para_nonlinear_start(:,reps),opts, ...
            share, x_rand, x_det, ind_mkt_start,ind_mkt_end, randomdraw,...
            IV, W, setup, n_oem, n_carrier, n_mkt, carrier_ind, oem_ind, [], []);
        trialest(:,reps) = para_nonlinear_est(:);
    catch err
        fprintf(pfid, 'error message = %s\n', err.message);
        for i = 1:length(err.stack)
            fprintf(pfid, '%s, Line %1.0f\n', err.stack(i).file, err.stack(i).line);
        end
        continue;
    end

    fprintf(pfid, '^^^^ done with stage-2 estimation: starting point %1.0f, time used = %1.0f ^^^^\n', reps, cputime - t0);
end

ind1 = find(trialflag==1);
if run_for_test
    ind1 = 1;
else
    if isempty(ind1); disp('stage-2 estimation did not converge. abort the program'); return; end;
end
[~, ind2] = min(trialfval(ind1));
ind = ind1(ind2);
optimal_est_second_stage = trialest(:,ind);
meanutility00 = trialmeanutility(:,ind);

[f, ~, ~, para_linear_est] = gmmobj_BLP_smartphone(optimal_est_second_stage, share, x_rand, x_det,ind_mkt_start,ind_mkt_end, randomdraw, IV, W, setup, n_oem, n_carrier, n_mkt, carrier_ind, oem_ind, meanutility00, []);

save('best_est_stage2.mat');
para_nonlinear_est = optimal_est_second_stage;

%% random coefficient estimation - standard error
[stderr, para_mean_chara, stderr_nonlinear_q, stderr_q] ...
    = gmm_std_err_smartphone(para_nonlinear_est, share,...
    x_rand, x_det, ind_mkt_start,ind_mkt_end, randomdraw, IV, W, ...
    setup, n_oem, n_carrier, n_mkt, carrier_ind, oem_ind, ...
    meanutility00, para_linear_est, ...
    pfid);

save('part1_demand_est')

%% Part 2. Marginal cost
pause(5)
fprintf(pfid, '^^^ starting master_est_smartphone_part2 ^^^^\n');
master_est_smartphone_part2

%% Part 3. Fixed cost
pause(5)
fprintf(pfid, '^^^ starting master_est_smartphone_part3 ^^^^\n');
master_est_smartphone_part3

%% counterfactual
pause(5)
fprintf(pfid, '^^^ starting counterfactual_smartphone ^^^^\n');
counterfactual_smartphone
